package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetVideoPlayClipsReferrerTimeseries queries Redshift to retrieves clip video-play referrer breakdowns
func (c *Client) GetVideoPlayClipsReferrerTimeseries(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayClipsReferrersTimeseriesUnit, error) {
	statement := `
        WITH modified_clips AS
        (   SELECT DISTINCT vod_id
                FROM tahoe_recent."video-play"
            WHERE   date >= $1
            AND     date < $2
            AND     time_utc >= $3
            AND     time_utc < $4
            AND     vod_type = 'clip'
            AND     vod_id IS NOT NULL
            AND     vod_id SIMILAR TO '[0-9]%'
        ), recent_clips AS
        (
            SELECT clip_id,
                date_trunc('microsec', date_trunc('hour', time_utc) + INTERVAL '5 min' * FLOOR(date_part('minute', time_utc) / 5.0)) AS create_time_utc,
                channel_id
            FROM tahoe_recent.create_clip
            WHERE   date >= $5
            AND     date < $2
        ), modified_groupings AS
        (
            SELECT DISTINCT b.channel_id,
            b.create_time_utc
            FROM modified_clips a
            INNER JOIN recent_clips b
            ON a.vod_id = b.clip_id
        ), grouping_referrers AS
        (
            SELECT b.clip_id,
                a.create_time_utc,
                c.referrer,
                sum(c.view_count) as views,
                a.channel_id
            FROM modified_groupings a
            INNER JOIN recent_clips b
            ON a.channel_id = b.channel_id
                AND a.create_time_utc = b.create_time_utc
            INNER JOIN analysis.clips c
            ON c.clip_id = b.clip_id
            GROUP BY a.channel_id, a.create_time_utc, b.clip_id, c.referrer
        )

        SELECT channel_id,
            create_time_utc,
            referrer,
            sum(views) AS views
        FROM grouping_referrers
        GROUP BY channel_id, create_time_utc, referrer
        ORDER BY channel_id ASC, create_time_utc DESC, referrer ASC
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	// We will only recalculate clip views if it was created within the last 7 days
	aceClipCutoff := aceStart.Truncate(day).AddDate(0, 0, -7)

	var results []VideoPlayClipsReferrersTimeseriesUnit
	rows, err := c.QueryContext(ctx, statement, aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat), start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceClipCutoff.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get clips video-play timeseries: query failed")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get clips video-play timeseries: failed to close rows"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit VideoPlayClipsReferrersTimeseriesUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.Timestamp,
			&unit.Referrer,
			&unit.Views,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get clips video-play timeseries: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
