package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetVideoPlayGeoSessionAggregate queries Redshift to retrieves geo session aggregates for
// the video_play spade event
func (c *Client) GetVideoPlayGeoSessionAggregate(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayGeoAggregateUnit, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >= $1
             AND     segment_start_time < $2
        ),   recent_video_plays AS
        (    SELECT  channel_id,
                     date_trunc('minute', time_utc) AS time_utc,
                     CASE
                          WHEN country = '' OR country IS NULL THEN 'unknown'
                          ELSE country
                     END AS country_code
             FROM    tahoe_recent."video-play"
             WHERE   date >= $3
             AND     date < $4
             AND     (live IS true OR live IS NULL)
        )

        SELECT a.channel_id,
               a.segment_start_time,
               b.country_code AS country,
               count(b.country_code) AS count
        FROM recent_sessions a
        INNER JOIN recent_video_plays b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time, b.country_code
        ORDER BY a.channel_id DESC, a.segment_start_time DESC
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []VideoPlayGeoAggregateUnit
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get geo video_play aggregates: failed to query video geo based count by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get geo video_play aggregates: failed to close rows when querying video geo based by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit VideoPlayGeoAggregateUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.SegmentStartTime,
			&unit.Country,
			&unit.Count,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get geo video_play aggregates: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
