package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetVideoPlayPlatformSessionAggregate queries Redshift to retrieves platform session aggregates for
// the video_play spade event
func (c *Client) GetVideoPlayPlatformSessionAggregate(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayPlatformAggregateUnit, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >= $1
             AND     segment_start_time < $2
        ),   recent_video_plays AS
        (    SELECT  channel_id,
                     date_trunc('minute', time_utc) AS time_utc,
                     CASE WHEN position('ps' IN player) > 0 OR position('ps' IN platform) > 0 OR position('xbox' IN platform) > 0 OR platform = 'playstation' THEN 'console'
                          WHEN player IN ('chromecast','chromecast-mediaplayer','roku','amazon','firetv','androidtv','firetv_stick') then 'tv'
                          WHEN platform ='web' THEN 'web'
                          WHEN platform ='android' THEN 'android'
                          WHEN platform ='mobile_web' THEN 'mobile_web'
                          WHEN platform ='ios' THEN 'ios'
                          WHEN platform ='curse' THEN 'desktop_app'
                          WHEN platform ='' or platform is null then 'unknown'
                          ELSE 'other'
                     END AS calculated_platform
             FROM    tahoe_recent."video-play"
             WHERE   date >= $3
             AND     date < $4
             AND     (live IS true OR live IS NULL)
        )

        SELECT a.channel_id,
               a.segment_start_time,
               b.calculated_platform AS platform,
               count(b.calculated_platform) AS count
        FROM recent_sessions a
        INNER JOIN recent_video_plays b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time, b.calculated_platform
        ORDER BY a.channel_id DESC, a.segment_start_time DESC
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []VideoPlayPlatformAggregateUnit
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get platform video_play aggregates: failed to query video platform based by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get platform video_play aggregates: failed to close rows when querying video platform based by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit VideoPlayPlatformAggregateUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.SegmentStartTime,
			&unit.Platform,
			&unit.Count,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get platform video_play aggregates: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
