package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetVideoPlayReferrerSessionAggregate queries Redshift to retrieves platform session aggregates for
// the video_play spade event
func (c *Client) GetVideoPlayReferrerSessionAggregate(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayReferrerAggregateUnit, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >= $1
             AND     segment_start_time < $2
        ),   recent_classified AS
        (    SELECT  channel_id,
                     date_trunc('minute', time_utc) AS time_utc,
                     platform AS platform,
                     medium AS medium,
                     content AS content,
                     player AS player,
                     referrer,
                     SPLIT_PART(referrer, '/', 3) AS referrer_domain,
                     REGEXP_SUBSTR(SPLIT_PART(referrer, '/', 4), '[^?#]*') AS referrer_path_root,
                     REGEXP_SUBSTR(SPLIT_PART(referrer, '/', 5), '[^?#]*') AS referrer_path_stem,
                     referrer_url,
                     SPLIT_PART(referrer_url, '/', 3) AS referrer_url_domain,
                     host_channel,
                     CASE
                        WHEN player <> 'embed'
                            AND ((host_channel <> '' AND host_channel IS NOT NULL)
                            OR position('twitch' IN referrer) > 0
                            OR content IS NOT NULL
                            OR medium IS NOT NULL
                            OR player = 'frontpage'
                            OR player = 'creative') THEN 'internal'
                        ELSE 'external'
                     END AS classification
             FROM    tahoe_recent."video-play"
             WHERE   date >= $3
             AND     date < $4
             AND     (live IS true OR live IS NULL)
             AND     platform = 'web'
        ),   recent_sorted AS
        (    SELECT channel_id,
                    time_utc,
                    classification,
                    CASE
                        WHEN classification='internal' THEN
                            CASE
                                WHEN player = 'frontpage' THEN 'front_page_featured'
                                WHEN medium = 'creative' THEN 'creative_page_featured'
                                WHEN host_channel IS NOT NULL AND host_channel > 0 THEN host_channel
                                WHEN medium = 'email' THEN 'email_live_notification'
                                WHEN medium = 'notification_toast' THEN 'onsite_notification'
                                WHEN medium = 'twitch_topnav' AND content = 'directory_following' THEN 'followed_channel'
                                WHEN medium = 'twitch_topnav' AND content = 'directory_games' THEN 'directory_browse'
                                WHEN medium = 'twitch_topnav' THEN 'top_nav_bar'
                                WHEN medium = 'twitch_home' AND content = 'carousel' THEN 'homepage'
                                WHEN medium = 'twitch_home' AND content = 'carousel_promo' THEN 'homepage_carousel'
                                WHEN medium = 'twitch_home' AND content = 'carousel_backfill' THEN 'homepage_carousel'
                                WHEN medium = 'twitch_home' AND content IN (
                                    'video_thumbnail',
                                    'video_title',
                                    'shelf_video_thumbnail',
                                    'shelf_video_title',
                                    'shelf_game_metadata') THEN 'homepage_recommendations'
                                WHEN medium = 'twitch_following' THEN 'followed_channel'
                                WHEN medium = 'twitch_socialcolumn' AND content = 'recommended_channels' THEN 'recommended_channel'
                                WHEN medium = 'twitch_socialcolumn' AND content = 'followed_channels' THEN 'followed_channel'
                                WHEN medium = 'twitch_directory' AND content = 'live_channel' THEN 'directory_browse'
                                WHEN medium = 'twitch_directory' AND content = 'language_stream' THEN 'directory_browse'
                                WHEN medium = 'following_directory' AND content = 'live_channel' THEN 'followed_channel'
                                WHEN medium = 'header_search' THEN 'search'
                                WHEN medium = 'clips_web' THEN 'clips_live'
                                WHEN medium IN ('twitch_games_directory', 'twitch_community_directory', 'twitch_browse_directory') THEN 'directory_browse'
                                WHEN medium = 'friend_presence' THEN 'friend_presence'
                                WHEN referrer_domain = 'www.twitch.tv' AND referrer_path_root = 'directory' AND referrer_path_stem = 'following' THEN 'followed_channel'
                                WHEN referrer_domain = 'www.twitch.tv' AND referrer_path_root = 'directory' THEN 'directory_browse'
                                WHEN referrer_domain = 'www.twitch.tv' AND referrer_path_root IN ('videos',
                                     'collections',
                                     'inventory',
                                     'bits',
                                     '404',
                                     'pr',
                                     'settings',
                                     'private',
                                     'create-landing',
                                     'communities',
                                     'friends',
                                     'extensions',
                                     'events',
                                     'streamplus',
                                     'streams',
                                     'youtuberedirect',
                                     'broadcast',
                                     'backpack',
                                     'email-verification',
                                     'index',
                                     'hi',
                                     'passport-callback',
                                     'team') THEN 'other'
                                WHEN referrer_domain = 'www.twitch.tv' AND referrer_path_root <> '' THEN referrer_path_root
                                ELSE 'other'
                            END
                        ELSE
                            CASE
                                WHEN (player = 'embed' AND referrer_url = '' OR referrer_url_domain = '' OR referrer_url = 'undefined' OR referrer_url IS NULL) THEN 'unknown'
                                WHEN player = 'embed' THEN referrer_url_domain
                                WHEN (referrer = '' OR referrer = 'undefined' OR referrer_domain = '' OR referrer IS NULL) THEN 'unknown'
                                ELSE referrer_domain
                            END
                    END AS source
             FROM    recent_classified
        )

        SELECT channel_id,
            segment_start_time,
            classification,
            source,
            count
        FROM (
           SELECT a.channel_id,
           a.segment_start_time,
           b.classification,
           b.source AS source,
           count(b.source) AS count,
           row_number() over (partition by a.channel_id, a.segment_start_time, b.classification order by count desc) as rank
           FROM recent_sessions a
           INNER JOIN recent_sorted b
           ON a.segment_start_time <= b.time_utc
               AND a.segment_end_time >= b.time_utc
               AND a.channel_id = b.channel_id
           GROUP BY a.channel_id, a.segment_start_time, b.classification, b.source
        )
        WHERE rank <= 100
        ORDER BY channel_id DESC, segment_start_time DESC, classification DESC, rank ASC
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []VideoPlayReferrerAggregateUnit
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get referrer video_play aggregates: failed to query video referrer based by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get referrer video_play aggregates: failed to close rows when querying video referrer based by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit VideoPlayReferrerAggregateUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.SegmentStartTime,
			&unit.Type,
			&unit.Source,
			&unit.Count,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get referrer video_play aggregates: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
