package redshift

import (
	"context"
	"fmt"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetVideoPlayUniqueSessionAggregateEndingAt queries Redshift to retrieves unique video-play aggregates
// for the video-play spade event
func (c *Client) GetVideoPlayUniqueSessionAggregateEndingAt(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayUniqueAggregateUnit, error) {
	return c.getVideoPlayUniqueSessionAggregateWithTimeKey(ctx, start, end, sessionsEndingAt)
}

// GetVideoPlayUniqueSessionAggregate queries Redshift to retrieves unique video-play aggregates
// for the video-play spade event
func (c *Client) GetVideoPlayUniqueSessionAggregate(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayUniqueAggregateUnit, error) {
	return c.getVideoPlayUniqueSessionAggregateWithTimeKey(ctx, start, end, sessionsStartingAt)
}

// GetVideoPlayUniqueSessionAggregate queries Redshift to retrieves unique video-play aggregates
// for the video-play spade event
func (c *Client) getVideoPlayUniqueSessionAggregateWithTimeKey(ctx context.Context, start time.Time, end time.Time, key string) ([]VideoPlayUniqueAggregateUnit, error) {
	statement := fmt.Sprintf(`
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   %v >= $1
             AND     %v < $2
        ),   recent_video_plays AS
        (    SELECT  channel_id,
                     device_id,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent."video-play"
             WHERE   date >= $3
             AND     date < $4
             AND     (live IS true OR live IS NULL)
        )

        SELECT a.channel_id,
               a.segment_start_time,
               count(DISTINCT b.device_id) AS unique_views,
               count(b.device_id) AS total_views
        FROM recent_sessions a
        INNER JOIN recent_video_plays b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time
        ORDER BY a.channel_id DESC, a.segment_start_time DESC
    `, key, key)

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []VideoPlayUniqueAggregateUnit
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get unique video_play aggregates: failed to query unique views count by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get video-play session: failed to close rows when querying unique views by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg VideoPlayUniqueAggregateUnit
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.UniqueViews,
			&agg.TotalViews,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get video-play session: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
