package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	sessionTemp = `
    CREATE TEMP TABLE purged_sessions
    DISTSTYLE ALL
    SORTKEY (4, 2)
    AS (
        SELECT is_segmented,
            segment_start_time,
            segment_end_time,
            channel_id
            FROM    analysis.sessions
        WHERE   segment_start_time >= $1
        AND     segment_start_time < $2
    )
    `

	sessionPurge = `
    DELETE
    FROM    analysis.sessions
    WHERE   segment_start_time >= $1
    AND     segment_start_time < $2;
    `

	sessionInsert = `
    INSERT INTO analysis.sessions
    WITH with_gaps AS
    (
             SELECT   channel_id ,
                      (date_trunc('minute', time_utc) - Lag(date_trunc('minute', time_utc), 1) OVER (partition BY channel_id ORDER BY time_utc ASC)) AS gap ,
                      date_trunc('minute', time_utc)                                                                                                 AS minute_trunc ,
                      time_utc
             FROM     tahoe_recent.minute_broadcast
             WHERE    date >= $1
             AND      date < $2) , with_sessions AS
    (
             SELECT   sum(
                      CASE
                               WHEN gap > interval '15 minutes' THEN 1
                               ELSE 0
                      END) OVER (partition BY channel_id ORDER BY time_utc ASC rows BETWEEN UNBOUNDED PRECEDING AND CURRENT row) AS session ,
                      channel_id ,
                      time_utc
             FROM     with_gaps ) , complete_sessions AS
    (
             SELECT   date_trunc('minute', min(time_utc)) AS start_time ,
                      date_trunc('minute', max(time_utc)) AS end_time ,
                      CASE
                               WHEN max(time_utc) - min(time_utc) <= interval '48 hours' THEN 1
                               ELSE extract(day from date_trunc('day', max(time_utc)) - date_trunc('day', min(time_utc)))
                      END AS chunks ,
                      channel_id
             FROM     with_sessions
             GROUP BY channel_id,
                      session ) , calculated_sessions AS
    (
              SELECT    chunks > 1                                        AS is_segmented ,
                        CASE
                                  WHEN chunks = 1 THEN end_time
                                  WHEN integers.INDEX = 1 THEN end_time
                                  ELSE date_trunc('day', end_time - (integers.INDEX - 2) * interval '1 days') - interval '1 minutes'
                        END AS segment_end_time ,
                        CASE
                                  WHEN chunks = 1 THEN start_time
                                  WHEN start_time > end_time - integers.INDEX * interval '1 days' THEN start_time
                                  ELSE date_trunc('day', end_time - (integers.INDEX - 1) * interval '1 days')
                        END AS segment_start_time ,
                        channel_id
              FROM      complete_sessions
              LEFT JOIN analysis.integer integers
              ON        complete_sessions.chunks >= integers.INDEX
    )
    SELECT * FROM calculated_sessions
    WHERE   segment_start_time >= $3
    AND     segment_start_time < $4;
    `

	sessionQuery = `
    SELECT  is_segmented,
            segment_start_time,
            segment_end_time,
            channel_id
    FROM    analysis.sessions
    WHERE   segment_start_time >= $1
    AND     segment_start_time < $2;
    `

	sessionDiff = `
    SELECT  is_segmented,
            segment_start_time,
            segment_end_time,
            channel_id
    FROM    purged_sessions a
    WHERE NOT EXISTS (
        SELECT 1
            FROM analysis.sessions b
        WHERE b.segment_start_time >= $1
        AND   b.segment_start_time < $2
        AND   a.channel_id = b.channel_id
        AND   a.segment_start_time = b.segment_start_time
    )
    `

	sessionTempPurge = `
    DROP TABLE IF EXISTS purged_sessions
    `
)

// CalculateSessions builds sessions from minute_broadcast events. And then inserts them into
// an intermediate table called analysis.sessions to calculate session aggregates
func (c *Client) CalculateSessions(ctx context.Context, start time.Time, end time.Time) ([]Session, []Session, error) {
	// we need to query outside the range that we want to generate sessions for to make sure sessions
	// on the end are/aren't 24 hr streams
	queryStart := start.Truncate(day).AddDate(0, 0, -2)
	queryEnd := end.Truncate(day).AddDate(0, 0, 3)

	var created []Session
	var purged []Session
	txn, err := c.BeginTx(ctx, nil)
	if err != nil {
		return nil, nil, errors.Wrap(err, "redshift calculate sessions: failed to start transaction")
	}

	defer func() {
		if err != nil {
			err = txn.Rollback()

		}
		err = txn.Commit()
	}()

	// build a temp containing previous sessions -- we will diff this vs
	// the new inserted sessions to determine which sessions we need to purge
	_, err = txn.ExecContext(ctx, sessionTemp, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, nil, err
	}

	// we are afraid of thie minute_broadcast table being improperly populated
	// so we will first remove the existing sessions we have
	_, err = txn.ExecContext(ctx, sessionPurge, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, nil, err
	}

	// we insert the newly calculated sessions. We need to query for an additional
	// 2 days of data because we dont have enough data (need at least 48hr+ of date) to ID
	// whether sessions are segmented (24hr stream) or not
	_, err = txn.ExecContext(ctx, sessionInsert, queryStart.Format(SQLTimeFormat), queryEnd.Format(SQLTimeFormat), start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, nil, err
	}

	// query for the sessions that we inserted to persist into dynamo
	createdRows, err := txn.QueryContext(ctx, sessionQuery, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, nil, err
	}

	defer func() {
		err = createdRows.Close()
		if err != nil {
			msg := "redshift calculate sessions: failed to close created rows"

			log.WithError(err).Error(msg)
		}
	}()

	for createdRows.Next() {
		var session Session
		err = createdRows.Scan(
			&session.IsSegmented,
			&session.SegmentStartTime,
			&session.SegmentEndTime,
			&session.ChannelID,
		)

		created = append(created, session)
	}

	err = createdRows.Err()
	if err != nil {
		msg := "redshift calculate session: error scanning created rows"

		log.WithError(err).Error(msg)
		return nil, nil, errors.Wrap(err, msg)
	}

	// this is to handle http://go-database-sql.org/surprises.html#multiple-statement-support
	err = createdRows.Close()
	if err != nil {
		msg := "redshift calculate session: failed to close created rows"

		log.WithError(err).Error(msg)
	}

	// query for the sessions that we need to delete from dynamo
	purgedRows, err := txn.QueryContext(ctx, sessionDiff, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, nil, err
	}

	defer func() {
		err = purgedRows.Close()
		if err != nil {
			msg := "redshift calculate sessions: failed to close purged rows"

			log.WithError(err).Error(msg)
		}
	}()

	for purgedRows.Next() {
		var session Session
		err = purgedRows.Scan(
			&session.IsSegmented,
			&session.SegmentStartTime,
			&session.SegmentEndTime,
			&session.ChannelID,
		)

		purged = append(purged, session)
	}

	err = purgedRows.Err()
	if err != nil {
		msg := "redshift calculate session: error scanning purged rows"

		log.WithError(err).Error(msg)
		return nil, nil, errors.Wrap(err, msg)
	}

	// this is to handle http://go-database-sql.org/surprises.html#multiple-statement-support
	err = purgedRows.Close()
	if err != nil {
		msg := "redshift calculate session: failed to close purged rows"

		log.WithError(err).Error(msg)
	}

	_, err = txn.ExecContext(ctx, sessionTempPurge)
	if err != nil {
		return nil, nil, err
	}

	return created, purged, nil
}
