package sqs

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"

	"code.justin.tv/cb/semki/internal/awscredentials"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

const (
	// MessageAttributeKeyName is the header we should look for to find
	// the stat name
	MessageAttributeKeyName = "Name"
	// MessageAttributeKeyTry is the header we should look for to find
	// the which try attempt this is
	MessageAttributeKeyTry = "Try"
	// MessageAttributeKeyType identifies what type of write operation we are
	// trying to do
	MessageAttributeKeyType = "Type"
	// SQSReplayMax is the most we will replay a batch put
	SQSReplayMax = 12
	// BatchSize are the amount of rows we will put in a batch put into sqs
	BatchSize = 10

	// TypeHeaderPut is what the type header will be for batch puts
	TypeHeaderPut = "PUT"
	// TypeHeaderDelete is what the type header will be for batch deletes
	TypeHeaderDelete = "DELETE"
)

// Client is a wrapper for the sqs.SQS client for a specific queue.
type Client struct {
	sqs      *sqs.SQS
	queueURL string
}

// NewClient creates an instance of an SQS Client
func NewClient(env string, region string, queueURL string) (*Client, error) {
	sess, err := session.NewSession(&aws.Config{
		Credentials: awscredentials.New(env, region),
		Region:      aws.String(region),
	})

	if err != nil {
		return nil, errors.Wrap(err, "sqs: failed to initialize client")
	}

	return &Client{
		sqs:      sqs.New(sess),
		queueURL: queueURL,
	}, nil
}

// Add sends a message to the specified topic
func (c *Client) Add(ctx context.Context, message Message) error {
	req, err := message.BuildInputObj()
	if err != nil {
		return err
	}

	req.SetQueueUrl(c.queueURL)
	if err = req.Validate(); err != nil {
		return errors.Wrap(err, fmt.Sprintf("sqs: failed to build message: %s", message.Message))
	}

	_, err = c.sqs.SendMessageWithContext(ctx, req)
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("sqs: failed to send message: %s", req.String()))
	}

	return nil
}

// AddBatch sends a group of messages to the specified topic
func (c *Client) AddBatch(ctx context.Context, messages []Message, batchName string) error {
	if len(messages) == 0 {
		return nil
	}

	entries := make([]*sqs.SendMessageBatchRequestEntry, len(messages))
	for idx := range messages {
		var err error
		entries[idx], err = messages[idx].BuildBatchEntryObj()
		if err != nil {
			return err
		}

		entries[idx].SetId(fmt.Sprintf("%s_%d", messages[idx].Name, idx))
		if err = entries[idx].Validate(); err != nil {
			return errors.Wrap(err, fmt.Sprintf("sqs: failed to build message: %s", messages[idx].Message))
		}
	}

	req := &sqs.SendMessageBatchInput{}

	req.SetEntries(entries)
	req.SetQueueUrl(c.queueURL)

	if err := req.Validate(); err != nil {
		return errors.Wrap(err, "sqs: failed to build input")
	}

	out, err := c.sqs.SendMessageBatchWithContext(ctx, req)
	if err != nil {
		return errors.Wrap(err, "sqs: failed to send batch")
	}

	for _, failedEntry := range out.Failed {
		log.WithFields(log.Fields{
			"stat":         batchName,
			"code":         *failedEntry.Code,
			"message":      *failedEntry.Message,
			"sender_fault": *failedEntry.SenderFault,
		}).Error("failed batch entry")
	}

	return nil
}

// BuildBatchEntryObj converts a Message obj into an aws SendMessageBatchRequestEntry
func (m *Message) BuildBatchEntryObj() (*sqs.SendMessageBatchRequestEntry, error) {
	try := 1
	if m.Retry != nil {
		try = *m.Retry
	}

	writeType := TypeHeaderPut
	if m.Type != nil {
		writeType = *m.Type
	}

	nameAttr, err := messageAttributeValueString(m.Name)
	if err != nil {
		msg := fmt.Sprintf("sqs: invalid message attribute value for %s", m.Name)
		return nil, errors.Wrap(err, msg)
	}

	tryAttr, err := messageAttributeValueNumber(try)
	if err != nil {
		msg := fmt.Sprintf("sqs: invalid message attribute value for %s", m.Name)
		return nil, errors.Wrap(err, msg)
	}

	typeAttr, err := messageAttributeValueString(writeType)
	if err != nil {
		msg := fmt.Sprintf("sqs: invalid message attribute value for %s", m.Name)
		return nil, errors.Wrap(err, msg)
	}

	messageString, err := jsonMarshalToString(m.Message)
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("sqs: failed to json marshal message: %s", m.Message))
	}

	messageAttributes := map[string]*sqs.MessageAttributeValue{
		MessageAttributeKeyName: nameAttr,
		MessageAttributeKeyTry:  tryAttr,
		MessageAttributeKeyType: typeAttr,
	}

	res := &sqs.SendMessageBatchRequestEntry{}
	res.SetMessageAttributes(messageAttributes)
	res.SetMessageBody(messageString)

	return res, nil
}

// BuildInputObj converts a Message obj into an aws SendMessageInput
func (m *Message) BuildInputObj() (*sqs.SendMessageInput, error) {
	try := 1
	if m.Retry != nil {
		try = *m.Retry
	}

	writeType := TypeHeaderPut
	if m.Type != nil {
		writeType = *m.Type
	}

	nameAttr, err := messageAttributeValueString(m.Name)
	if err != nil {
		msg := fmt.Sprintf("sqs: invalid message attribute value for %s", m.Name)
		return nil, errors.Wrap(err, msg)
	}

	tryAttr, err := messageAttributeValueNumber(try)
	if err != nil {
		msg := fmt.Sprintf("sqs: invalid message attribute value for %s", m.Name)
		return nil, errors.Wrap(err, msg)
	}

	typeAttr, err := messageAttributeValueString(writeType)
	if err != nil {
		msg := fmt.Sprintf("sqs: invalid message attribute value for %s", m.Name)
		return nil, errors.Wrap(err, msg)
	}

	messageString, err := jsonMarshalToString(m.Message)
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("sqs: failed to json marshal message: %s", m.Message))
	}

	messageAttributes := map[string]*sqs.MessageAttributeValue{
		MessageAttributeKeyName: nameAttr,
		MessageAttributeKeyTry:  tryAttr,
		MessageAttributeKeyType: typeAttr,
	}

	res := &sqs.SendMessageInput{}
	res.SetMessageAttributes(messageAttributes)
	res.SetMessageBody(messageString)

	return res, nil
}

const messageAttributeValueDataTypeString = "String"

func messageAttributeValueString(str string) (*sqs.MessageAttributeValue, error) {
	attribute := &sqs.MessageAttributeValue{}

	attribute.SetDataType(messageAttributeValueDataTypeString)
	attribute.SetStringValue(str)

	if err := attribute.Validate(); err != nil {
		return nil, err
	}

	return attribute, nil
}

const messageAttributeValueDataTypeNumber = "Number"

func messageAttributeValueNumber(val int) (*sqs.MessageAttributeValue, error) {
	attribute := &sqs.MessageAttributeValue{}

	attribute.SetDataType(messageAttributeValueDataTypeNumber)
	attribute.SetStringValue(strconv.Itoa(val))

	if err := attribute.Validate(); err != nil {
		return nil, err
	}

	return attribute, nil
}

func jsonMarshalToString(message interface{}) (string, error) {
	messageBytes, err := json.Marshal(message)
	if err != nil {
		return "", err
	}

	return string(messageBytes), nil
}
