package statsd

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	log "github.com/sirupsen/logrus"
)

type responseWriter struct {
	http.ResponseWriter
	status int
}

const sampleRate = 1.0

// StatsLogger is an HTTP middleware that logs the request.
func (c *Client) StatsLogger(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, r *http.Request) {
		start := time.Now().UTC()
		logged := &responseWriter{
			ResponseWriter: w,
		}

		inner.ServeHTTP(logged, r)

		duration := time.Since(start)
		formattedRequestURI := strings.Replace(strings.Trim(r.RequestURI, "/"), "/", ".", -1)
		statName := fmt.Sprintf("request.%s.%d", formattedRequestURI, logged.status)

		go func() {
			err := c.TimingDuration(statName, duration, sampleRate)
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"stat_name": statName,
					"duration":  duration,
				}).Warn("statsd: failed to send timing duration")
			}
		}()
	}

	return http.HandlerFunc(middleware)
}

func (rw *responseWriter) WriteHeader(status int) {
	rw.status = status
	rw.ResponseWriter.WriteHeader(status)
}
