package cron

import (
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/cb/semki/config"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/stats"
	"code.justin.tv/cb/semki/internal/stats/broadcast_id_per_session"
	"code.justin.tv/cb/semki/internal/stats/chat_per_session"
	"code.justin.tv/cb/semki/internal/stats/chat_timeseries"
	"code.justin.tv/cb/semki/internal/stats/clips"
	"code.justin.tv/cb/semki/internal/stats/commercial_per_session"
	"code.justin.tv/cb/semki/internal/stats/commercial_timeseries"
	"code.justin.tv/cb/semki/internal/stats/concurrents_per_session"
	"code.justin.tv/cb/semki/internal/stats/concurrents_timeseries"
	"code.justin.tv/cb/semki/internal/stats/minute_watched_host_raid_per_session"
	"code.justin.tv/cb/semki/internal/stats/raid_execute_per_session"
	"code.justin.tv/cb/semki/internal/stats/raid_execute_timeseries"
	"code.justin.tv/cb/semki/internal/stats/server_follow_per_session"
	"code.justin.tv/cb/semki/internal/stats/server_follow_timeseries"
	"code.justin.tv/cb/semki/internal/stats/sessions"
	"code.justin.tv/cb/semki/internal/stats/subscriptions_purchase_success_per_session"
	"code.justin.tv/cb/semki/internal/stats/subscriptions_purchase_success_timeseries"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_create_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_create_timeseries"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_referrer_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_referrer_timeseries"
	"code.justin.tv/cb/semki/internal/stats/video_play_geo_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_platform_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_referrer_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_unique_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_unique_timeseries"

	log "github.com/sirupsen/logrus"
)

const (
	nameHeaderAttr = "X-Aws-Sqsd-Attr-" + sqs.MessageAttributeKeyName
)

// calculate is the handler for calculating stats
func (s *Server) calculate(w http.ResponseWriter, req *http.Request) {
	var stat stats.Stat
	var end time.Time
	var start time.Time
	var defaultRewriteHours int
	name := req.Header.Get(nameHeaderAttr)
	if len(name) == 0 {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	startStr := req.Header.Get(startDateHeaderAttr)
	endStr := req.Header.Get(endDateHeaderAttr)
	if len(endStr) == 0 {
		end = time.Now().UTC().Add(time.Hour * 1).Truncate(hour)
	} else {
		parsed, err := time.Parse(time.RFC3339, endStr)
		if err != nil {
			log.Errorf("cron %s: bad backfill end time", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}

		end = parsed.UTC().Truncate(hour)
	}

	statClients := stats.Clients{
		Experiments: s.experiments,
		SQS:         s.sqs.ingest,
		Redshift:    s.redshift,
		Statsd:      s.statsd.ingestSQS,
		Pool:        s.pool,
	}

	switch name {
	case chattimeseries.Name:
		stat = chattimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = chattimeseries.RewriteHours
	case concurrentstimeseries.Name:
		stat = concurrentstimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = concurrentstimeseries.RewriteHours
	case serverfollowtimeseries.Name:
		stat = serverfollowtimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = serverfollowtimeseries.RewriteHours
	case broadcastidsession.Name:
		stat = broadcastidsession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = broadcastidsession.RewriteHours
	case chatsession.Name:
		stat = chatsession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = chatsession.RewriteHours
	case concurrentssession.Name:
		stat = concurrentssession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = concurrentssession.RewriteHours
	case serverfollowsession.Name:
		stat = serverfollowsession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = serverfollowsession.RewriteHours
	case sessions.Name:
		log.WithFields(log.Fields{
			"request": req,
		}).Info("calling sessions calc in events endpoint")

		stat = sessions.InitStat(&statClients, config.Environment)
		defaultRewriteHours = sessions.RewriteHours
	case clips.Name:
		log.WithFields(log.Fields{
			"request": req,
		}).Info("calling clips calc in events endpoint")

		stat = clips.InitStat(&statClients, config.Environment)
		defaultRewriteHours = clips.RewriteHours
	case videoplayclipsreferrersession.Name:
		stat = videoplayclipsreferrersession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayclipsreferrersession.RewriteHours
	case videoplayclipscreatesession.Name:
		stat = videoplayclipscreatesession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayclipscreatesession.RewriteHours
	case videoplayreferrersession.Name:
		stat = videoplayreferrersession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayreferrersession.RewriteHours
	case videoplaygeosession.Name:
		stat = videoplaygeosession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplaygeosession.RewriteHours
	case videoplayplatformsession.Name:
		stat = videoplayplatformsession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayplatformsession.RewriteHours
	case videoplayuniquesession.Name:
		stat = videoplayuniquesession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayuniquesession.RewriteHours
	case videoplayclipsreferrertimeseries.Name:
		stat = videoplayclipsreferrertimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayclipsreferrertimeseries.RewriteHours
	case videoplayclipscreatetimeseries.Name:
		stat = videoplayclipscreatetimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayclipscreatetimeseries.RewriteHours
	case videoplayuniquetimeseries.Name:
		stat = videoplayuniquetimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = videoplayuniquetimeseries.RewriteHours
	case subscriptionspurchasesuccessssession.Name:
		stat = subscriptionspurchasesuccessssession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = subscriptionspurchasesuccessssession.RewriteHours
	case subscriptionspurchasesuccesstimeseries.Name:
		stat = subscriptionspurchasesuccesstimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = subscriptionspurchasesuccesstimeseries.RewriteHours
	case commercialsession.Name:
		stat = commercialsession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = commercialsession.RewriteHours
	case commercialtimeseries.Name:
		stat = commercialtimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = commercialtimeseries.RewriteHours
	case raidexecutesession.Name:
		stat = raidexecutesession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = raidexecutesession.RewriteHours
	case raidexecutetimeseries.Name:
		stat = raidexecutetimeseries.InitStat(&statClients, config.Environment)
		defaultRewriteHours = raidexecutetimeseries.RewriteHours
	case minutewatchedhostraidedsession.Name:
		stat = minutewatchedhostraidedsession.InitStat(&statClients, config.Environment)
		defaultRewriteHours = minutewatchedhostraidedsession.RewriteHours
	default:
		log.Errorf("cron %s: name provided did not match any known stat", name)

		w.WriteHeader(http.StatusBadRequest)
		return
	}

	if len(startStr) == 0 {
		// use the default rewrite value to calculate a start time if one isnt
		//  provided
		start = end.Add(time.Hour * time.Duration(-1*defaultRewriteHours)).Truncate(hour)
	} else {
		parsed, err := time.Parse(time.RFC3339, startStr)
		if err != nil {
			log.Errorf("cron %s: bad backfill start time", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}

		start = parsed.UTC().Truncate(hour)
	}

	// send tracking for each completed progression
	log.Infof("cron: calculating %s from %s to %s", name, start, end)
	calcStart := time.Now()

	err := stat.Calculate(req.Context(), start, end)
	if err != nil {
		log.WithFields(log.Fields{
			"stat":  name,
			"start": start,
			"end":   end,
		}).WithError(err)

		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	go func(statName string, calcStart time.Time) {
		err = s.statsd.cronSQS.Dec(fmt.Sprintf("%s.messages", statName), 1, 1)
		if err != nil {
			log.WithError(err).Error("failed to send stat to sqs statsd")
		}

		err = s.statsd.cronSvc.TimingDuration(fmt.Sprintf("calc.success.%s", statName), time.Since(calcStart), 1)
		if err != nil {
			log.WithError(err).Error("failed to send timing to svc statsd")
		}
	}(name, calcStart)

	w.WriteHeader(http.StatusOK)
}
