package cron

import (
	"net/http"
	"time"

	"goji.io"
	"goji.io/pat"
	"golang.org/x/sync/semaphore"

	"code.justin.tv/cb/semki/internal/clients/experiments"
	"code.justin.tv/cb/semki/internal/clients/redshift"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/clients/statsd"
	"code.justin.tv/cb/semki/internal/httputil"
)

const (
	startDateHeaderAttr = "X-Aws-Sqsd-Attr-Backfill-Start-Date"
	endDateHeaderAttr   = "X-Aws-Sqsd-Attr-Backfill-End-Date"
	cronFrequency       = 45 // This is based on the cron job's frequency as defined in cron.yaml
	day                 = 24 * time.Hour
	hour                = 1 * time.Hour
)

// Server contains a router and client interfaces to downstream services.
type Server struct {
	mux                *goji.Mux
	experiments        *experiments.Client
	sqs                SQSClients
	statsd             StatsdClients
	redshift           redshift.Redshift
	pool               *semaphore.Weighted
	backfillInProgress bool
}

// ServerParams contains the required components for instantiating a Server.
type ServerParams struct {
	CronSQS            sqs.SQS
	IngestSQS          sqs.SQS
	Redshift           redshift.Redshift
	CronSvcStatsd      *statsd.Client
	CronSQSStatsd      *statsd.Client
	Experiments        *experiments.Client
	IngestSQSStatsd    *statsd.Client
	Pool               *semaphore.Weighted
	BackfillInProgress bool
}

// SQSClients is a collection of sqs clients to be used by the cron service
type SQSClients struct {
	cron   sqs.SQS
	ingest sqs.SQS
}

// StatsdClients is a collection of statsd clients used to measure service state
type StatsdClients struct {
	cronSvc   *statsd.Client
	cronSQS   *statsd.Client
	ingestSQS *statsd.Client
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:         goji.NewMux(),
		experiments: params.Experiments,
		sqs: SQSClients{
			cron:   params.CronSQS,
			ingest: params.IngestSQS,
		},
		statsd: StatsdClients{
			cronSvc:   params.CronSvcStatsd,
			cronSQS:   params.CronSQSStatsd,
			ingestSQS: params.IngestSQSStatsd,
		},
		redshift:           params.Redshift,
		pool:               params.Pool,
		backfillInProgress: params.BackfillInProgress,
	}

	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)

	root.Use(httputil.PanicRecoveryMiddleware)
	root.Use(server.statsd.cronSvc.StatsLogger)

	root.HandleFunc(pat.Post("/calc/sessions"), server.sessions)
	root.HandleFunc(pat.Post("/calc/event"), server.calculate)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
