package cron

import (
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/cb/semki/config"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/stats"
	"code.justin.tv/cb/semki/internal/stats/clips"
	"code.justin.tv/cb/semki/internal/stats/emails"
	"code.justin.tv/cb/semki/internal/stats/sessions"

	log "github.com/sirupsen/logrus"
	"golang.org/x/sync/errgroup"
)

// sessions
func (s *Server) sessions(w http.ResponseWriter, req *http.Request) {
	if s.backfillInProgress {
		log.Info("cron: cron is disabled due to backfilling, returning")
		return
	}
	statClients := stats.Clients{
		Experiments: s.experiments,
		SQS:         s.sqs.ingest,
		Redshift:    s.redshift,
		Statsd:      s.statsd.ingestSQS,
		Pool:        s.pool,
	}

	group, ctx := errgroup.WithContext(req.Context())
	// Generate sessions
	group.Go(func() error {
		end := time.Now().UTC().Add(time.Hour * 1).Truncate(hour)
		start := end.Add(time.Hour * time.Duration(-1*sessions.RewriteHours)).Truncate(hour)

		log.Infof("Calculating sessions for %s to %s", start, end)
		stat := sessions.InitStat(&statClients, config.Environment)
		createdSessions, err := stat.CalculateAndReturnSessions(ctx, start, end)
		log.Infof("Finished calculating sessions for %s to %s", start, end)
		if err != nil {
			return err
		}

		// Generate stream summary emails if new sessions were successfully created
		end = time.Now().UTC().Add(time.Minute * time.Duration(-1*cronFrequency))
		start = end.Add(time.Minute * time.Duration(-2*cronFrequency))

		log.Infof("Calculating email stats for %s to %s", start, end)
		email := emails.InitStat(&statClients)
		err = email.Calculate(ctx, start, end, createdSessions)
		log.Infof("Finished calculating email stats for %s to %s", start, end)

		return err
	})

	// Generate clips
	group.Go(func() error {
		end := time.Now().UTC().Truncate(day).AddDate(0, 0, 1)
		start := end.Add(time.Hour * time.Duration(-1*clips.RewriteHours)).Truncate(day)

		log.Infof("Calculating clips for %s to %s", start, end)
		stat := clips.InitStat(&statClients, config.Environment)
		err := stat.Calculate(ctx, start, end)
		log.Infof("Finished calculating clips for %s to %s", start, end)

		return err
	})

	if err := group.Wait(); err != nil {
		log.WithError(err).Error("cron: failed to calculate sessions or clips")
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	for _, stat := range config.Values.Stats {
		err := s.sqs.cron.Add(req.Context(), sqs.Message{
			Name:    stat,
			Message: struct{}{},
			Retry:   nil,
		})
		if err != nil {
			log.WithError(err).Errorf("cron: failed to send to SQS for %s", stat)
			continue
		}

		err = s.statsd.cronSQS.Inc(fmt.Sprintf("%s.messages", stat), 1, 1)
		if err != nil {
			log.WithError(err).Errorf("failed to send stat to StatsD for %s", stat)
		}
	}

	w.WriteHeader(http.StatusOK)
}
