package ingest

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/cb/semki/config"
	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"code.justin.tv/cb/semki/internal/stats/concurrents_per_session"
	"code.justin.tv/cb/semki/internal/stats/emails"
	"code.justin.tv/cb/semki/internal/stats/server_follow_per_session"
	"code.justin.tv/cb/semki/internal/stats/sessions"
	"code.justin.tv/cb/semki/internal/stats/subscriptions_purchase_success_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_referrer_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_unique_per_session"
	"code.justin.tv/chat/pushy/client/events"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

func (s *Server) sendEmail(ctx context.Context, session emails.SessionData) error {
	sessionStart := session.SegmentStartTime.UnixNano() / (int64(time.Millisecond) / int64(time.Nanosecond))
	sessionEnd := session.SegmentEndTime.UnixNano() / (int64(time.Millisecond) / int64(time.Nanosecond))
	streamSummaryLink := fmt.Sprintf(
		"/dashboard/stream-summary/%v-%v",
		sessionStart,
		sessionEnd,
	)

	streamSummaryEvent := events.StreamSummaryParams{
		UserID:            session.ChannelID,
		StreamSummaryLink: streamSummaryLink,
	}

	// Current stream stats
	streamData, err := s.getSessionStats(ctx, session.SegmentID)
	if err != nil {
		return err
	}
	streamSummaryEvent.RecentStream = streamData
	streamSummaryEvent.RecentStream.StreamDuration = session.SegmentEndTime.Sub(session.SegmentStartTime)

	// Previous stream stats
	previousSegmentID, err := s.getPreviousSessionID(ctx, session.ChannelID, session.SegmentStartTime)
	if err == nil && len(previousSegmentID) > 0 {
		previousStreamData, prevErr := s.getSessionStats(ctx, previousSegmentID)
		if prevErr == nil {
			streamSummaryEvent.PreviousStream = previousStreamData
		}
	}

	// Channel referrer data
	referrer, referrerCount, err := s.getTopChannelReferrer(ctx, session)
	if err == nil {
		streamSummaryEvent.Snapshot.TopReferrerLogin = referrer
		streamSummaryEvent.Snapshot.TopReferrerViews = referrerCount
	}

	// Quest data
	questData, err := s.achievements.GetQuestData(ctx, session.ChannelID)
	if err == nil {
		streamSummaryEvent.Quest = questData
	}

	// Clips data
	topClip, err := s.clips.GetTopClipFromStream(ctx, session.ChannelID, session.SegmentStartTime, session.SegmentEndTime)
	if err == nil && topClip != nil {
		streamSummaryEvent.Snapshot.TopClipImageURL = topClip.Thumbnails["small"]
		streamSummaryEvent.Snapshot.TopClipTitle = topClip.Title
		streamSummaryEvent.Snapshot.TopClipURL = topClip.URL
	}

	return s.pushy.Publish(ctx, events.StreamSummaryEvent, streamSummaryEvent)
}

func (s *Server) getSessionStats(ctx context.Context, segmentID string) (events.StreamData, error) {
	env := config.Environment
	streamData := events.StreamData{}

	out, err := s.dynamoDB.BatchGet(ctx, &dynamodb.BatchGetItemInput{
		RequestItems: map[string]*dynamodb.KeysAndAttributes{
			concurrentssession.GetTableName(env): {
				Keys: []map[string]*dynamodb.AttributeValue{
					{
						"segment_id": &dynamodb.AttributeValue{
							S: aws.String(segmentID),
						},
					},
				},
			},
			serverfollowsession.GetTableName(env): {
				Keys: []map[string]*dynamodb.AttributeValue{
					{
						"segment_id": &dynamodb.AttributeValue{
							S: aws.String(segmentID),
						},
					},
				},
			},
			subscriptionspurchasesuccessssession.GetTableName(env): {
				Keys: []map[string]*dynamodb.AttributeValue{
					{
						"segment_id": &dynamodb.AttributeValue{
							S: aws.String(segmentID),
						},
					},
				},
			},
			videoplayuniquesession.GetTableName(env): {
				Keys: []map[string]*dynamodb.AttributeValue{
					{
						"segment_id": &dynamodb.AttributeValue{
							S: aws.String(segmentID),
						},
					},
				},
			},
		},
	})
	if err != nil {
		return streamData, err
	}

	// Read concurrents table
	resp := out.Responses[concurrentssession.GetTableName(env)]
	if resp != nil && len(resp) > 0 {
		averageConcurrents, err := strconv.ParseFloat(*resp[0]["average_concurrents"].N, 64)
		if err == nil {
			streamData.AverageViewers = averageConcurrents
		}

		maxConcurrents, err := strconv.ParseInt(*resp[0]["max_concurrents"].N, 10, 64)
		if err == nil {
			streamData.PeakViewers = maxConcurrents
		}
	}

	// Read follows table
	resp = out.Responses[serverfollowsession.GetTableName(env)]
	if resp != nil && len(resp) > 0 {
		newFollowers, err := strconv.ParseInt(*resp[0]["new_followers"].N, 10, 64)
		if err == nil {
			streamData.NewFollowers = newFollowers
		}
	}

	// Read subscriptions table
	resp = out.Responses[subscriptionspurchasesuccessssession.GetTableName(env)]
	if resp != nil && len(resp) > 0 {
		newSubscriptions, err := strconv.ParseInt(*resp[0]["new_subs"].N, 10, 64)
		if err == nil {
			streamData.NewSubscriptions = newSubscriptions
		}
	}

	// Read uniques table
	resp = out.Responses[videoplayuniquesession.GetTableName(env)]
	if resp != nil && len(resp) > 0 {
		uniqueViews, err := strconv.ParseInt(*resp[0]["unique_views"].N, 10, 64)
		if err == nil {
			streamData.UniqueViewers = uniqueViews
		}
	}

	return streamData, nil
}

func (s *Server) getPreviousSessionID(ctx context.Context, channelID string, startTime time.Time) (string, error) {
	env := config.Environment

	out, err := s.dynamoDB.Query(ctx, &dynamodb.QueryInput{
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":ch_id": {
				S: aws.String(channelID),
			},
			":start_time": {
				S: aws.String(startTime.Format(dynamo.DynamoTimeFormat)),
			},
		},
		KeyConditionExpression: aws.String("channel_id = :ch_id AND segment_start_time < :start_time"),
		Limit:            aws.Int64(1),
		ScanIndexForward: aws.Bool(false),
		TableName:        aws.String(sessions.GetTableName(env)),
	})
	if err != nil {
		return "", err
	}

	if len(out.Items) == 0 {
		return "", nil
	}

	return *out.Items[0]["segment_id"].S, nil
}

var siteReferrers = map[string]bool{
	"other":                    true,
	"front_page_featured":      true,
	"creative_page_featured":   true,
	"hosted":                   true,
	"email_live_notification":  true,
	"onsite_notification":      true,
	"followed_channel":         true,
	"directory_browse":         true,
	"search":                   true,
	"clips_live":               true,
	"friend_presence":          true,
	"top_nav_bar":              true,
	"recommended_channel":      true,
	"homepage_carousel":        true,
	"homepage_recommendations": true,
	"other_channel_page":       true,
}

func (s *Server) getTopChannelReferrer(ctx context.Context, session emails.SessionData) (string, int64, error) {
	env := config.Environment

	user, err := s.users.GetUserByID(ctx, session.ChannelID, nil)
	if err != nil || user == nil || user.Login == nil {
		return "", 0, err
	}

	out, err := s.dynamoDB.BatchGet(ctx, &dynamodb.BatchGetItemInput{
		RequestItems: map[string]*dynamodb.KeysAndAttributes{
			videoplayreferrersession.GetTableName(env): {
				Keys: []map[string]*dynamodb.AttributeValue{
					{
						"segment_id": &dynamodb.AttributeValue{
							S: aws.String(session.SegmentID),
						},
					},
				},
			},
		},
	})
	if err != nil {
		return "", 0, err
	}

	// Read concurrents table
	resp := out.Responses[videoplayreferrersession.GetTableName(env)]
	if resp == nil || len(resp) == 0 {
		return "", 0, nil
	}

	internalReferrers := resp[0]["internal_referrers"].M
	topReferrer := ""
	topReferrerCount := int64(0)

	for referrer, referrerValue := range internalReferrers {
		if !siteReferrers[referrer] {
			referrerCount, err := strconv.ParseInt(*referrerValue.N, 10, 64)
			if err == nil && referrer != *user.Login && referrerCount > topReferrerCount {
				topReferrer = referrer
				topReferrerCount = referrerCount
			}
		}
	}

	return topReferrer, topReferrerCount, nil
}
