package ingest

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"math"
	"math/rand"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/cb/semki/config"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/stats/broadcast_id_per_session"
	"code.justin.tv/cb/semki/internal/stats/chat_per_session"
	"code.justin.tv/cb/semki/internal/stats/chat_timeseries"
	"code.justin.tv/cb/semki/internal/stats/commercial_per_session"
	"code.justin.tv/cb/semki/internal/stats/commercial_timeseries"
	"code.justin.tv/cb/semki/internal/stats/concurrents_per_session"
	"code.justin.tv/cb/semki/internal/stats/concurrents_timeseries"
	"code.justin.tv/cb/semki/internal/stats/emails"
	"code.justin.tv/cb/semki/internal/stats/minute_watched_host_raid_per_session"
	"code.justin.tv/cb/semki/internal/stats/raid_execute_per_session"
	"code.justin.tv/cb/semki/internal/stats/raid_execute_timeseries"
	"code.justin.tv/cb/semki/internal/stats/server_follow_per_session"
	"code.justin.tv/cb/semki/internal/stats/server_follow_timeseries"
	"code.justin.tv/cb/semki/internal/stats/sessions"
	"code.justin.tv/cb/semki/internal/stats/subscriptions_purchase_success_per_session"
	"code.justin.tv/cb/semki/internal/stats/subscriptions_purchase_success_timeseries"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_create_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_create_timeseries"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_referrer_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_clips_referrer_timeseries"
	"code.justin.tv/cb/semki/internal/stats/video_play_geo_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_platform_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_referrer_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_unique_per_session"
	"code.justin.tv/cb/semki/internal/stats/video_play_unique_timeseries"

	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	log "github.com/sirupsen/logrus"
)

const (
	nameHeaderAttr     = "X-Aws-Sqsd-Attr-" + sqs.MessageAttributeKeyName
	tryCountHeaderAttr = "X-Aws-Sqsd-Attr-" + sqs.MessageAttributeKeyTry
	typeHeaderAttr     = "X-Aws-Sqsd-Attr-" + sqs.MessageAttributeKeyType

	retryDelay  = 60
	backoffCap  = 180
	backoffInit = 60
	backoffUnit = 5
)

// ingest
func (s *Server) ingest(w http.ResponseWriter, req *http.Request) {
	var collection interface{}
	var puts []*dynamodb.WriteRequest
	var table string
	name := req.Header.Get(nameHeaderAttr)
	if len(name) == 0 {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	try := req.Header.Get(tryCountHeaderAttr)
	if len(try) == 0 {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	batchType := req.Header.Get(typeHeaderAttr)
	if len(batchType) == 0 || (batchType != sqs.TypeHeaderPut && batchType != sqs.TypeHeaderDelete) {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	tryInt, err := strconv.Atoi(try)
	if err != nil {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	body, err := ioutil.ReadAll(req.Body)
	if err != nil {
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	switch name {
	case sessions.Name:
		table = sessions.GetTableName(config.Environment)
		parsed := sessions.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		switch batchType {
		case sqs.TypeHeaderPut:
			puts, err = parsed.CreateDynamoPuts(req.Context())
		case sqs.TypeHeaderDelete:
			puts, err = parsed.CreateDynamoDeletes(req.Context())
		}
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case broadcastidsession.Name:
		table = broadcastidsession.GetTableName(config.Environment)
		parsed := broadcastidsession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case chatsession.Name:
		table = chatsession.GetTableName(config.Environment)
		parsed := chatsession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case chattimeseries.Name:
		table = chattimeseries.GetTableName(config.Environment)
		parsed := chattimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case concurrentstimeseries.Name:
		table = concurrentstimeseries.GetTableName(config.Environment)
		parsed := concurrentstimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case concurrentssession.Name:
		table = concurrentssession.GetTableName(config.Environment)
		parsed := concurrentssession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case serverfollowsession.Name:
		table = serverfollowsession.GetTableName(config.Environment)
		parsed := serverfollowsession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case serverfollowtimeseries.Name:
		table = serverfollowtimeseries.GetTableName(config.Environment)
		parsed := serverfollowtimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayreferrersession.Name:
		table = videoplayreferrersession.GetTableName(config.Environment)
		parsed := videoplayreferrersession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplaygeosession.Name:
		table = videoplaygeosession.GetTableName(config.Environment)
		parsed := videoplaygeosession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayplatformsession.Name:
		table = videoplayplatformsession.GetTableName(config.Environment)
		parsed := videoplayplatformsession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayuniquesession.Name:
		table = videoplayuniquesession.GetTableName(config.Environment)
		parsed := videoplayuniquesession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayuniquetimeseries.Name:
		table = videoplayuniquetimeseries.GetTableName(config.Environment)
		parsed := videoplayuniquetimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayclipscreatetimeseries.Name:
		table = videoplayclipscreatetimeseries.GetTableName(config.Environment)
		parsed := videoplayclipscreatetimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayclipscreatesession.Name:
		table = videoplayclipscreatesession.GetTableName(config.Environment)
		parsed := videoplayclipscreatesession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayclipsreferrersession.Name:
		table = videoplayclipsreferrersession.GetTableName(config.Environment)
		parsed := videoplayclipsreferrersession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case videoplayclipsreferrertimeseries.Name:
		table = videoplayclipsreferrertimeseries.GetTableName(config.Environment)
		parsed := videoplayclipsreferrertimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case subscriptionspurchasesuccessssession.Name:
		table = subscriptionspurchasesuccessssession.GetTableName(config.Environment)
		parsed := subscriptionspurchasesuccessssession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case subscriptionspurchasesuccesstimeseries.Name:
		table = subscriptionspurchasesuccesstimeseries.GetTableName(config.Environment)
		parsed := subscriptionspurchasesuccesstimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case commercialsession.Name:
		table = commercialsession.GetTableName(config.Environment)
		parsed := commercialsession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case commercialtimeseries.Name:
		table = commercialtimeseries.GetTableName(config.Environment)
		parsed := commercialtimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case raidexecutesession.Name:
		table = raidexecutesession.GetTableName(config.Environment)
		parsed := raidexecutesession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())
		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case raidexecutetimeseries.Name:
		table = raidexecutetimeseries.GetTableName(config.Environment)
		parsed := raidexecutetimeseries.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())

		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case minutewatchedhostraidedsession.Name:
		table = minutewatchedhostraidedsession.GetTableName(config.Environment)
		parsed := minutewatchedhostraidedsession.Collection{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		collection = parsed

		puts, err = parsed.CreateDynamoPuts(req.Context())

		if err != nil {
			log.WithError(err).Errorf("ingest %s: failed to marshal puts", name)

			w.WriteHeader(http.StatusBadRequest)
			return
		}
	case emails.Name:
		parsed := emails.SessionData{}
		if err = json.Unmarshal(body, &parsed); err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}

		err = s.sendEmail(req.Context(), parsed)
		if err != nil {
			w.WriteHeader(http.StatusBadRequest)
			return
		}

		w.WriteHeader(http.StatusOK)
		return
	default:
		log.Errorf("ingest %s: name provided did not match any known stat", name)

		w.WriteHeader(http.StatusBadRequest)
		return
	}

	go func() {
		ingestStart := time.Now()

		backoff := rand.Intn(int(math.Min(backoffCap, float64(backoffInit+(tryInt-1)*backoffUnit*2))))
		err = s.dynamoDB.BatchPut(context.Background(), puts, table, backoff)
		if err != nil {
			if tryInt == sqs.SQSReplayMax {
				if aerr, ok := err.(awserr.Error); ok {
					if aerr.Code() != dynamodb.ErrCodeProvisionedThroughputExceededException {
						log.WithError(err).WithFields(log.Fields{
							"stat":  name,
							"table": table,
							"rows":  puts,
						}).Errorf("ingest %s: failed to save to dynamo", name)
					}
				}

				s.onDynamoMessageFailure(name)
				return
			}

			s.onDynamoCycleFailure(name)
			time.Sleep(time.Duration(retryDelay) * time.Second)
			newTry := tryInt + 1

			err := s.sqs.Add(context.Background(), sqs.Message{
				Name:    name,
				Message: collection,
				Retry:   &newTry,
			})
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"stat":  name,
					"table": table,
					"rows":  puts,
				}).Errorf("ingest %s: failed to retry", name)
			}
			return
		}

		s.onDynamoSuccess(name, ingestStart)
	}()

	w.WriteHeader(http.StatusOK)
	return
}

func (s *Server) onDynamoSuccess(name string, ingestStart time.Time) {
	err := s.statsd.ingestSQS.Dec(fmt.Sprintf("%s.messages", name), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}

	err = s.statsd.ingestSvc.Inc(fmt.Sprintf("status.%d.%s", http.StatusOK, name), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}

	err = s.statsd.ingestSvc.TimingDuration(fmt.Sprintf("ingest.success.%s", name), time.Since(ingestStart), 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}

func (s *Server) onDynamoCycleFailure(name string) {
	err := s.statsd.ingestSvc.Inc(fmt.Sprintf("status.%d.%s", http.StatusInternalServerError, name), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}

func (s *Server) onDynamoMessageFailure(name string) {
	err := s.statsd.ingestSQS.Dec(fmt.Sprintf("%s.messages", name), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}
