package ingest

import (
	"net/http"

	"goji.io"
	"goji.io/pat"

	"code.justin.tv/cb/semki/internal/clients/achievements"
	"code.justin.tv/cb/semki/internal/clients/clips"
	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/clients/statsd"
	"code.justin.tv/cb/semki/internal/httputil"
	"code.justin.tv/chat/pushy/client/events"
	users "code.justin.tv/web/users-service/client/usersclient_internal"
)

// Server contains a router and client interfaces to downstream services.
type Server struct {
	sqs          sqs.SQS
	mux          *goji.Mux
	achievements achievements.Achievements
	clips        clips.Clips
	dynamoDB     dynamo.Dynamo
	pushy        events.Publisher
	statsd       StatsdClients
	users        users.InternalClient
}

// ServerParams contains the required components for instantiating a Server.
type ServerParams struct {
	IngestSQS       sqs.SQS
	Achievements    achievements.Achievements
	Clips           clips.Clips
	DynamoDB        dynamo.Dynamo
	Pushy           events.Publisher
	IngestSvcStatsd *statsd.Client
	IngestSQSStatsd *statsd.Client
	Users           users.InternalClient
}

// StatsdClients is a collection of statsd clients used to measure service state
type StatsdClients struct {
	ingestSvc *statsd.Client
	ingestSQS *statsd.Client
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:          goji.NewMux(),
		achievements: params.Achievements,
		clips:        params.Clips,
		dynamoDB:     params.DynamoDB,
		pushy:        params.Pushy,
		sqs:          params.IngestSQS,
		statsd: StatsdClients{
			ingestSvc: params.IngestSvcStatsd,
			ingestSQS: params.IngestSQSStatsd,
		},
		users: params.Users,
	}

	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(httputil.PanicRecoveryMiddleware)
	root.Use(server.statsd.ingestSvc.StatsLogger)

	root.HandleFunc(pat.Post("/ingest"), server.ingest)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
