package concurrentssession

import (
	"context"
	"fmt"

	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

// DynamoRow is how a session is represented in dynamoDB
type DynamoRow struct {
	SegmentID           string           `dynamodbav:"segment_id"`
	MaxConcurrents      int64            `dynamodbav:"max_concurrents,int"`
	AverageConcurrents  float64          `dynamodbav:"average_concurrents,float"`
	TotalMinutesWatched int64            `dynamodbav:"total_minutes_watched,int"`
	PeakCCUTime         dynamo.Timestamp `dynamodbav:"peak_concurrents_time"`
}

// Collection is a collection of session DynamoRows
type Collection []*DynamoRow

// CreateDynamoPuts takes a list of Dynamo rows and converts them to a list of puts
func (rows Collection) CreateDynamoPuts(ctx context.Context) ([]*dynamodb.WriteRequest, error) {
	result := make([]*dynamodb.WriteRequest, len(rows))

	for idx, row := range rows {
		marshaled, err := dynamodbattribute.MarshalMap(row)
		if err != nil {
			msg := fmt.Sprintf("stat %s: failed to marshal row", Name)

			log.WithError(err).Error(msg)
			return []*dynamodb.WriteRequest{}, errors.Wrap(err, msg)
		}

		result[idx] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: marshaled,
			},
		}
	}

	return result, nil
}
