package concurrentstimeseries

import (
	"context"
	"fmt"
	"math"
	"time"

	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/stats"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	// Name is the unique name of this stat
	Name = "concurrentstimeseries"
	// TableFormat is the name of the dynamo table used as persistent storage
	TableFormat = "cb-semki-%s-channel-concurrents-timeseries"
	// RewriteHours is the minimum amount of hours of data we need to overwrite
	// timeseries arent dependent on sessions and are really expensive Dynamo-wise
	RewriteHours = 6 // 6 hours
)

// Stat contains the clients the stat needs
type Stat struct {
	Clients *stats.Clients
	Env     string
}

// InitStat prepares a stat for calculation
func InitStat(clients *stats.Clients, env string) *Stat {
	return &Stat{
		Clients: clients,
		Env:     env,
	}
}

// GetTableName returns the dynamo table name for this stat
func GetTableName(env string) string {
	return fmt.Sprintf(TableFormat, env)
}

// Calculate calculates average ccu and sends results to dynamo
func (s *Stat) Calculate(ctx context.Context, start time.Time, end time.Time) error {
	sessions, err := s.Clients.Redshift.GetConcurrentsTimeseries(ctx, start, end)
	if err != nil {
		msg := fmt.Sprintf("stat %s: redshift query failed", Name)

		log.WithError(err).Error(msg)
		return errors.Wrap(err, msg)
	}

	totalSQSBatches := int(math.Ceil(float64(len(sessions)) / float64(dynamo.BatchSize) / float64(sqs.BatchSize)))
	for sqsBatch := 0; sqsBatch < totalSQSBatches; sqsBatch++ {
		start := sqsBatch * dynamo.BatchSize * sqs.BatchSize
		end := int(math.Min(float64(start+dynamo.BatchSize*sqs.BatchSize), float64(len(sessions))))
		totalDynBatches := int(math.Ceil(float64(end-start) / float64(dynamo.BatchSize)))
		SQSbatch := make([]sqs.Message, totalDynBatches)
		for dynBatch := 0; dynBatch < totalDynBatches; dynBatch++ {
			dynStart := start + dynamo.BatchSize*dynBatch
			dynEnd := int(math.Min(float64(dynStart)+float64(dynamo.BatchSize), float64(end)))

			batch := make([]*DynamoRow, dynEnd-dynStart)
			for idx, result := range sessions[dynStart:dynEnd] {
				batch[idx] = &DynamoRow{
					ChannelID:          result.ChannelID,
					Timestamp:          dynamo.Timestamp{Converted: result.Timestamp},
					AverageConcurrents: result.AverageConcurrents,
				}
			}

			SQSbatch[dynBatch] = sqs.Message{
				Name:    Name,
				Message: batch,
				Retry:   nil,
			}
		}

		if err := s.Clients.Pool.Acquire(ctx, 1); err != nil {
			msg := fmt.Sprintf("stat %s: failed to acquire semaphore", Name)

			log.WithError(err).Error(msg)
			return errors.Wrap(err, msg)
		}

		go s.Clients.SendSQSMessage(SQSbatch, Name, sqsBatch, totalDynBatches)
	}

	return nil
}
