package emails

import (
	"context"
	"fmt"
	"math"
	"time"

	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"code.justin.tv/cb/semki/internal/clients/redshift"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/stats"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	// Name is the unique name of this stat
	Name = "stream-summary-emails"
	// minSessionLengthForEmail is the minimum number of minutes needed in a session to send a stream summary email
	minSessionLengthForEmail = 10
)

// Stat contains the clients the stat needs
type Stat struct {
	Clients *stats.Clients
}

// InitStat prepares a stat for calculation
func InitStat(clients *stats.Clients) *Stat {
	return &Stat{
		Clients: clients,
	}
}

// Calculate calculates sessions and sends results to dynamo
func (s *Stat) Calculate(ctx context.Context, start time.Time, end time.Time, sessions []redshift.Session) error {
	if len(sessions) == 0 {
		return nil
	}

	// Gather sessions that need a stream summary email
	emailSessions := []redshift.Session{}
	for _, session := range sessions {
		sessionIsInTimeFrame := session.SegmentEndTime.After(start) && session.SegmentEndTime.Before(end)
		sessionIsLongEnough := session.SegmentEndTime.Sub(session.SegmentStartTime).Minutes() >= minSessionLengthForEmail

		if sessionIsInTimeFrame && sessionIsLongEnough {
			channelIsInEmailExperiment, err := s.Clients.Experiments.IsInStreamSummaryExperiment(ctx, session.ChannelID)
			if err != nil {
				log.Errorf("ingest: error when checking experiment, %v", err)
			}
			log.Infof("email channel_id=%v start=%v end=%v experiment=%v", session.ChannelID, session.SegmentStartTime, session.SegmentEndTime, channelIsInEmailExperiment)

			if channelIsInEmailExperiment {
				emailSessions = append(emailSessions, session)
			}
		}
	}

	err := s.sendEmailsToIngest(ctx, emailSessions, Name)
	if err != nil {
		msg := fmt.Sprintf("stat %s: failed to send create batch", Name)

		log.WithError(err).Error(msg)
		return errors.Wrap(err, msg)
	}

	return nil
}

func (s *Stat) sendEmailsToIngest(ctx context.Context, sessions []redshift.Session, event string) error {
	totalSQSBatches := int(math.Ceil(float64(len(sessions)) / float64(sqs.BatchSize)))
	for sqsBatch := 0; sqsBatch < totalSQSBatches; sqsBatch++ {
		start := sqsBatch * sqs.BatchSize
		end := int(math.Min(float64(start+sqs.BatchSize), float64(len(sessions))))
		batchSize := end - start

		SQSbatch := make([]sqs.Message, batchSize)
		for i := 0; i < batchSize; i++ {
			session := sessions[i+start]
			SQSbatch[i] = sqs.Message{
				Name: event,
				Message: SessionData{
					ChannelID:        session.ChannelID,
					SegmentID:        fmt.Sprintf("%s:%s", session.ChannelID, session.SegmentStartTime.Format(dynamo.DynamoTimeFormat)),
					SegmentStartTime: session.SegmentStartTime,
					SegmentEndTime:   session.SegmentEndTime,
				},
			}
		}

		if err := s.Clients.Pool.Acquire(ctx, 1); err != nil {
			return err
		}

		go s.Clients.SendSQSMessage(SQSbatch, event, sqsBatch, batchSize)
	}

	return nil
}
