package raidexecutetimeseries

import (
	"context"
	"fmt"

	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

// DynamoRow is how a raid execution stats are represented in dynamoDB
type DynamoRow struct {
	TargetID      string           `dynamodbav:"target_id"`
	Timestamp     dynamo.Timestamp `dynamodbav:"timestamp"`
	NumberOfRaids int64            `dynamodbav:"number_of_raids,int"`
	ViewerCount   int64            `dynamodbav:"viewer_count,int"`
}

// Collection is a collection of raid execution stats DynamoRows
type Collection []*DynamoRow

// CreateDynamoPuts takes a list of Dynamo rows and converts them to a list of puts
func (rows Collection) CreateDynamoPuts(ctx context.Context) ([]*dynamodb.WriteRequest, error) {
	result := make([]*dynamodb.WriteRequest, len(rows))

	for idx, row := range rows {
		marshaled, err := dynamodbattribute.MarshalMap(row)
		if err != nil {
			msg := fmt.Sprintf("stat %s: failed to marshal row", Name)

			log.WithError(err).Error(msg)
			return []*dynamodb.WriteRequest{}, errors.Wrap(err, msg)
		}

		result[idx] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: marshaled,
			},
		}
	}

	return result, nil
}
