package sessions

import (
	"context"
	"fmt"

	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

// DynamoRow is how a session is represented in dynamoDB
type DynamoRow struct {
	SegmentID        string           `dynamodbav:"segment_id"`
	ChannelID        string           `dynamodbav:"channel_id"`
	SegmentStartTime dynamo.Timestamp `dynamodbav:"segment_start_time"`
	SegmentEndTime   dynamo.Timestamp `dynamodbav:"segment_end_time"`
	IsSegmented      bool             `dynamodbav:"is_segmented"`
}

// Collection is a collection of session DynamoRows
type Collection []*DynamoRow

// CreateDynamoPuts takes a list of Dynamo rows and converts them to a list of puts
func (rows Collection) CreateDynamoPuts(ctx context.Context) ([]*dynamodb.WriteRequest, error) {
	result := make([]*dynamodb.WriteRequest, len(rows))

	for idx, row := range rows {
		marshaled, err := dynamodbattribute.MarshalMap(row)
		if err != nil {
			msg := fmt.Sprintf("stat %s: failed to marshal row", Name)

			log.WithError(err).Error(msg)
			return []*dynamodb.WriteRequest{}, errors.Wrap(err, msg)
		}

		result[idx] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: marshaled,
			},
		}
	}

	return result, nil
}

// CreateDynamoDeletes takes a list of Dynamo rows and converts them into a list of deletes
func (rows Collection) CreateDynamoDeletes(ctx context.Context) ([]*dynamodb.WriteRequest, error) {
	result := make([]*dynamodb.WriteRequest, len(rows))

	for idx, row := range rows {
		result[idx] = &dynamodb.WriteRequest{
			DeleteRequest: &dynamodb.DeleteRequest{
				Key: map[string]*dynamodb.AttributeValue{
					"channel_id":         {S: aws.String(row.ChannelID)},
					"segment_start_time": {S: aws.String(row.SegmentStartTime.Converted.Format(dynamo.DynamoTimeFormat))},
				},
			},
		}
	}

	return result, nil
}
