package sessions

import (
	"context"
	"fmt"
	"math"
	"time"

	"code.justin.tv/cb/semki/config"
	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"code.justin.tv/cb/semki/internal/clients/redshift"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/stats"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	// Name is the unique name of this stat
	Name = "sessions"
	// TableFormat is the name of the dynamo table used as persistent storage
	TableFormat = "cb-semki-%s-sessions"
	// RewriteHours is the minimum amount of hours of sessions we need to overwrite
	// this is because we could have potentially changed a <48hr stream into 3 <=24hr streams
	RewriteHours = 4 * 24 // 4 days
)

// Stat contains the clients the stat needs
type Stat struct {
	Clients *stats.Clients
	Env     string
}

// InitStat prepares a stat for calculation
func InitStat(clients *stats.Clients, env string) *Stat {
	return &Stat{
		Clients: clients,
		Env:     env,
	}
}

// GetTableName returns the dynamo table name for this stat
func GetTableName(env string) string {
	return fmt.Sprintf(TableFormat, env)
}

// Calculate calculates sessions and sends results to dynamo
func (s *Stat) Calculate(ctx context.Context, start time.Time, end time.Time) error {
	_, err := s.CalculateAndReturnSessions(ctx, start, end)
	return err
}

// CalculateAndReturnSessions calculates sessions, sends results to dynamo, and returns the created sessions
func (s *Stat) CalculateAndReturnSessions(ctx context.Context, start time.Time, end time.Time) ([]redshift.Session, error) {
	var created []redshift.Session
	var deleted []redshift.Session
	var err error

	putBatch := sqs.TypeHeaderPut
	deleteBatch := sqs.TypeHeaderDelete

	if s.Env == config.Production {
		created, deleted, err = s.Clients.Redshift.CalculateSessions(ctx, start, end)
	}
	// TODO cam: uncomment this if we want to do anything on staging with these sessions
	// else {
	// created, err = s.Clients.Redshift.GetRecentSessions(ctx, start, end)
	// }

	if err != nil {
		msg := fmt.Sprintf("stat %s: redshift query failed", Name)

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	err = s.sendSessionsToIngest(ctx, created, Name, &putBatch)
	if err != nil {
		msg := fmt.Sprintf("stat %s: failed to send create batch", Name)

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	err = s.sendSessionsToIngest(ctx, deleted, Name, &deleteBatch)
	if err != nil {
		msg := fmt.Sprintf("stat %s: failed to send delete batch", Name)

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return created, nil
}

func (s *Stat) sendSessionsToIngest(ctx context.Context, sessions []redshift.Session, event string, batchType *string) error {
	totalSQSBatches := int(math.Ceil(float64(len(sessions)) / float64(dynamo.BatchSize) / float64(sqs.BatchSize)))
	for sqsBatch := 0; sqsBatch < totalSQSBatches; sqsBatch++ {
		start := sqsBatch * dynamo.BatchSize * sqs.BatchSize
		end := int(math.Min(float64(start+dynamo.BatchSize*sqs.BatchSize), float64(len(sessions))))
		totalDynBatches := int(math.Ceil(float64(end-start) / float64(dynamo.BatchSize)))
		SQSbatch := make([]sqs.Message, totalDynBatches)
		for dynBatch := 0; dynBatch < totalDynBatches; dynBatch++ {
			dynStart := start + dynamo.BatchSize*dynBatch
			dynEnd := int(math.Min(float64(dynStart)+float64(dynamo.BatchSize), float64(end)))

			batch := make([]*DynamoRow, dynEnd-dynStart)
			for idx, result := range sessions[dynStart:dynEnd] {
				batch[idx] = &DynamoRow{
					SegmentID:        fmt.Sprintf("%s:%s", result.ChannelID, result.SegmentStartTime.Format(dynamo.DynamoTimeFormat)),
					ChannelID:        result.ChannelID,
					SegmentStartTime: dynamo.Timestamp{Converted: result.SegmentStartTime},
					SegmentEndTime:   dynamo.Timestamp{Converted: result.SegmentEndTime},
					IsSegmented:      result.IsSegmented,
				}
			}

			SQSbatch[dynBatch] = sqs.Message{
				Name:    event,
				Message: batch,
				Retry:   nil,
				Type:    batchType,
			}
		}

		if err := s.Clients.Pool.Acquire(ctx, 1); err != nil {
			return err
		}

		go s.Clients.SendSQSMessage(SQSbatch, event, sqsBatch, totalDynBatches)
	}

	return nil
}
