package stats

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"time"

	"code.justin.tv/cb/semki/internal/clients/experiments"
	"code.justin.tv/cb/semki/internal/clients/redshift"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/clients/statsd"

	log "github.com/sirupsen/logrus"
	"golang.org/x/sync/semaphore"
)

// Clients contains all client interfaces a stat requires to create dynamo rows
type Clients struct {
	Experiments *experiments.Client
	SQS         sqs.SQS
	Redshift    redshift.Redshift
	Statsd      *statsd.Client
	Pool        *semaphore.Weighted
}

// Stat represents the interface for ea statistic
// start, end represent the time range in which we want to calculate rows for
type Stat interface {
	Calculate(ctx context.Context, start time.Time, end time.Time) error
}

func getHostname() string {
	var hostname string
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown"
	}
	return hostname
}

// OnSemAcquire helps track the worker pool on the ec2 inst
func (c *Clients) OnSemAcquire() {
	err := c.Statsd.Inc(fmt.Sprintf("host.%s.semaphore", getHostname()), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}

// OnSemRelease helps track the worker pool on the ec2 inst
func (c *Clients) OnSemRelease() {
	err := c.Statsd.Dec(fmt.Sprintf("host.%s.semaphore", getHostname()), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}

// OnSQSSuccess contains statsd series to update on SQS post success
func (c *Clients) OnSQSSuccess(name string, length int) {
	err := c.Statsd.Inc(fmt.Sprintf("%s.messages", name), int64(length), 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}

	err = c.Statsd.Inc(fmt.Sprintf("status.%d", http.StatusOK), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}

// OnSQSFailure contains statsd series to update on SQS post failure
func (c *Clients) OnSQSFailure(name string) {
	err := c.Statsd.Inc(fmt.Sprintf("status.%d", http.StatusInternalServerError), 1, 1)
	if err != nil {
		log.WithError(err).Error("failed to send stat to statsd")
	}
}

// SendSQSMessage takes a batch of sqs messages and sends them
func (c *Clients) SendSQSMessage(messages []sqs.Message, name string, batch int, count int) {
	c.OnSemAcquire()

	defer func() {
		c.Pool.Release(1)
		c.OnSemRelease()
	}()

	err := c.SQS.AddBatch(context.Background(), messages, name)
	if err != nil {
		log.WithFields(log.Fields{
			"stat":  name,
			"batch": batch,
		}).WithError(err).Error("stats: failed to add batch to sqs")

		c.OnSQSFailure(name)
	} else {
		c.OnSQSSuccess(name, count)
	}
}
