// Package experiments provides a way to assign requests to groups to run experiments.
package experiments

import (
	"crypto/sha1"
	"encoding/binary"
	"math"
	"sync"
	"time"

	"code.justin.tv/feeds/spade"
)

// Experiments describes an experiment framework
type Experiments interface {
	// RegisterDefault add a default value for an experiment.
	RegisterDefault(kind Type, experimentName, experimentID, defaultGroup string)
	// RegisterOverride stores a local only override for a particular experiment and ids
	RegisterOverride(kind Type, experimentName, experimentID, group string, ids ...string) error
	// Treat buckets an identifier to an experiment group.
	// This function is deterministic: as long as the experiment does not change, calling it with the same parameters will
	// always return the same result.
	Treat(kind Type, experimentName, experimentID, id string) (string, error)
	// Close stops any goroutines created by the Experiments instance.
	Close() error
}

// Provider provides access to experiments stored in a datasource.
type Provider interface {
	// GetExperiment returns the Experiment that corresponds to the given name, or nil if the Experiment is
	// not defined in the datasource.
	GetExperiment(experimentID string) *Experiment

	// Close signals to the provider that it should stop any background work.
	Close()
}

// Config defines the configuration info needed by the default Experiments implementation.
//
// The Config interface allows us to configure experiments using fixed values now, and will allow us to integrate
// with configuration frameworks (like distconf) in the future.
type Config struct {
	// Platform describes which platform should be used to source experiments from
	Platform ExperimentPlatform

	// SpadeEndpoint is the URL to the Spade server that tracking events should be sent to.
	SpadeEndpoint string

	SpadeBacklogSize int64

	// DisableSpadeTracking disables sending tracking events to Spade.
	DisableSpadeTracking bool

	// PollingInterval is the amount of time to wait before re-fetching experiments from
	// GetSpadeEndpoint. Default is DefaultPollingInterval
	PollingInterval time.Duration

	// ErrorHandler is a function that is run on background errors that occur when fetching experiments and
	// sending tracking events.
	ErrorHandler func(error)

	// Override for the Tracking Client (most likely Spade) used to track experiment_branch events
	TrackingClient Tracking
}

// New returns the default implementation of Experiments that supports treating users into groups based on
// experiments defined locally, and also experiments defined in minixperiment.
func New(config Config) (Experiments, error) {
	if len(config.Platform) == 0 {
		config.Platform = LegacyPlatform
	}

	if config.SpadeEndpoint == "" {
		config.SpadeEndpoint = "https://spade.internal.justin.tv"
	}

	if config.SpadeBacklogSize <= 0 {
		config.SpadeBacklogSize = 1024
	}

	if config.PollingInterval <= 0 {
		config.PollingInterval = DefaultPollingInterval
	}

	var trackingClient Tracking
	if config.DisableSpadeTracking {
		trackingClient = noopTrackingClient{}
	} else if config.TrackingClient != nil {
		trackingClient = config.TrackingClient
	} else {
		trackingClient = NewTrackingClient(config)
	}

	mxpProvider, err := newMinixperimentProvider(config.Platform, config.PollingInterval, config.ErrorHandler)
	if err != nil {
		return nil, err
	}

	return &experimentImpl{
		defaults:  map[string]experimentDefault{},
		overrides: map[string]map[string]string{},
		tracking:  trackingClient,
		providers: []Provider{
			mxpProvider,
		},
	}, nil
}

// Dependencies are dependencies that can be injected into the default Experiments implementation.
type Dependencies struct {
	TrackingClient Tracking
	Providers      []Provider
}

// NewWithDependencies sets up an instance of Experiments by injecting the given dependencies.  It is meant to
// facilitate testing, and for consumers to set up an Experiments instance with custom Experiment providers.
func NewWithDependencies(deps *Dependencies) Experiments {
	return newWithDependencies(deps)
}

// newWithDependencies is the internal version of NewWithDependencies
func newWithDependencies(deps *Dependencies) *experimentImpl {
	return &experimentImpl{
		defaults:  map[string]experimentDefault{},
		overrides: map[string]map[string]string{},
		tracking:  deps.TrackingClient,
		providers: deps.Providers,
	}
}

type experimentImpl struct {
	// Map of experiment names to defaults
	defaults      map[string]experimentDefault
	defaultsLock  sync.RWMutex
	overrides     map[string]map[string]string
	overridesLock sync.RWMutex

	tracking  Tracking
	providers []Provider
}

// Close stops any goroutines created by the Experiments instance.
func (T *experimentImpl) Close() error {
	for _, provider := range T.providers {
		provider.Close()
	}

	return T.tracking.Close()
}

// RegisterDefault add a default value for an experiment.
func (T *experimentImpl) RegisterDefault(kind Type, experimentName, experimentID, defaultGroup string) {
	T.defaultsLock.Lock()
	defer T.defaultsLock.Unlock()

	T.defaults[experimentID] = experimentDefault{
		id:           experimentID,
		kind:         kind,
		defaultGroup: defaultGroup,
	}
}

// RegisterOverride add an override group for an experiment and id
func (T *experimentImpl) RegisterOverride(kind Type, experimentName, experimentID, group string, ids ...string) error {
	T.defaultsLock.Lock()
	def, ok := T.defaults[experimentID]
	T.defaultsLock.Unlock()
	if !ok {
		return ErrNoDefault
	}
	if def.kind != kind {
		return ErrInvalidType
	}

	T.overridesLock.Lock()
	defer T.overridesLock.Unlock()
	_, ok = T.overrides[experimentID]
	if !ok {
		T.overrides[experimentID] = map[string]string{}
	}

	for _, id := range ids {
		T.overrides[experimentID][id] = group
	}
	return nil
}

// experimentIsValid checks whether an experiment can be used
func experimentIsValid(exp Experiment) bool {
	if len(exp.ID) == 0 || (exp.Kind != UserType && exp.Kind != DeviceIDType) || len(exp.Name) == 0 || len(exp.Groups) == 0 {
		return false
	}

	sum := 0.0
	for _, g := range exp.Groups {
		if g.Value == "" {
			return false
		}
		sum += g.Weight
	}

	return sum > 0
}

// Treat buckets an identifier to an experiment group.
// This function is deterministic: as long as the experiment does not change, calling it with the same parameters will
// always return the same result.
func (T *experimentImpl) Treat(kind Type, experimentName, experimentID, id string) (string, error) {
	T.defaultsLock.RLock()
	def, defaultFound := T.defaults[experimentID]
	T.defaultsLock.RUnlock()
	if !defaultFound {
		return "", ErrNoDefault
	}

	T.overridesLock.RLock()
	overrides := T.overrides[experimentID]
	if overrides != nil {
		group, ok := overrides[id]
		if ok {
			T.overridesLock.RUnlock()
			return group, nil
		}
	}
	T.overridesLock.RUnlock()

	exp := T.getExperiment(experimentID)
	if exp == nil {
		if kind != def.kind {
			return "", ErrInvalidType
		}

		// Experiment was defaulted but not initialized (minixperiment unavailable or something), use default
		// group
		T.trackExperimentBranch(kind, experimentID, experimentName, id, def.defaultGroup)
		return def.defaultGroup, nil
	}

	if kind != exp.Kind {
		return "", ErrInvalidType
	}

	if g, err := exp.selectTreatment(id); err == nil {
		T.trackExperimentBranch(kind, exp.ID, exp.Name, id, g)
		return g, nil
	}

	// Should never reach this point under normal operation
	T.trackExperimentBranch(kind, experimentID, experimentName, id, def.defaultGroup)
	return def.defaultGroup, nil
}

func (T *experimentImpl) getExperiment(experimentID string) *Experiment {
	for _, provider := range T.providers {
		if exp := provider.GetExperiment(experimentID); exp != nil {
			return exp
		}
	}

	return nil
}

func (exp *Experiment) selectTreatment(id string) (string, error) {
	// Hash the experiment ID and device/user ID to a value in the range [0, 1).
	fraction := experimentHash(exp.ID, id, exp.Salt)

	total := 0.0
	for _, g := range exp.Groups {
		total += g.Weight
	}

	for _, g := range exp.Groups {
		fraction -= (g.Weight / total)
		if fraction <= 0 {
			return g.Value, nil
		}
	}

	return "", ErrOverRan
}

// trackExperimentBranch sends an experiment_branch tracking event to Spade.
func (T *experimentImpl) trackExperimentBranch(kind Type, experimentID, experimentName, userOrDeviceID, group string) {
	var experimentType string
	var userID string
	var deviceID string

	switch kind {
	case DeviceIDType:
		experimentType = "device_id"
		deviceID = userOrDeviceID
	case UserType:
		experimentType = "user_id"
		userID = userOrDeviceID
		// Use userID as deviceID to alleviate science performance issues
		deviceID = userOrDeviceID
	}

	T.tracking.QueueEvents(spade.Event{
		Name: "experiment_branch",
		Properties: &experimentBranchTrackingEvent{
			Group:          group,
			ExperimentID:   experimentID,
			ExperimentType: experimentType,
			ExperimentName: experimentName,
			UserID:         userID,
			DeviceID:       deviceID,
			Platform:       "backend-client",
		},
	})
}

// experimentHash converts an experimentID and id pair into a float64 in the range [0, 1).
// This replicates the algorithm used by minixperiment-client-js so that requests are assigned to same groups.
func experimentHash(experimentID, id, salt string) float64 {
	seed := experimentID + id + salt

	// Hash the experiment ID and user/device ID using SHA1.
	hashBytes := sha1.Sum([]byte(seed))

	// Get the first 32 bits of the hash value as an unsigned int.
	hashUint32 := binary.BigEndian.Uint32(hashBytes[:4])

	// Convert to floating point representation.
	hashFloat := float64(hashUint32)

	// Transform to a fraction between [0, 1).
	return hashFloat / float64(math.MaxUint32+1)
}
