package experiments

import (
	"context"
	"encoding/json"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"

	"code.justin.tv/feeds/errors"
)

const (
	getExperimentsErrorMessage = "getting experiments failed"
	minixperimentBucket        = "minixperiment-prod"
)

type minixperimentClient interface {
	getExperiments(ctx context.Context) ([]*Experiment, error)
}

type minixperimentClientImpl struct {
	s3       s3Client
	platform ExperimentPlatform
}

var _ minixperimentClient = &minixperimentClientImpl{}

func newS3MinixperimentClient(platform ExperimentPlatform) (*minixperimentClientImpl, error) {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})
	if err != nil {
		return nil, err
	}

	client := s3.New(sess)

	return &minixperimentClientImpl{
		s3:       client,
		platform: platform,
	}, nil
}

type s3Client interface {
	GetObjectWithContext(ctx aws.Context, input *s3.GetObjectInput, opts ...request.Option) (*s3.GetObjectOutput, error)
}

type minixpExperiment struct {
	Name   string  `json:"name"`
	Kind   Type    `json:"t"`
	Salt   int     `json:"s"`
	Groups []Group `json:"groups"`
}

func (c *minixperimentClientImpl) getExperiments(ctx context.Context) ([]*Experiment, error) {
	obj, err := c.s3.GetObjectWithContext(ctx, &s3.GetObjectInput{
		Bucket: aws.String(minixperimentBucket),
		Key:    aws.String(string(c.platform)),
	})
	if err != nil {
		return nil, err
	}

	var idToExperiment map[string]*minixpExperiment
	decoder := json.NewDecoder(obj.Body)
	err = decoder.Decode(&idToExperiment)
	if err != nil {
		return nil, errors.Wrap(err, getExperimentsErrorMessage)
	}

	exps := make([]*Experiment, 0, len(idToExperiment))
	for id, minixp := range idToExperiment {
		if minixp == nil {
			continue
		}

		var salt string
		if minixp.Salt != 0 {
			salt = fmt.Sprintf("%d", minixp.Salt)
		}

		exps = append(exps, &Experiment{
			ID:     id,
			Name:   minixp.Name,
			Kind:   minixp.Kind,
			Salt:   salt,
			Groups: minixp.Groups,
		})
	}
	return exps, nil
}
