package experiments

import "fmt"

// Type represents an experiment type
type Type int

var (
	// DeviceIDType represents an experiment on device_id
	DeviceIDType = Type(1)
	// UserType represents an experiment on user_id
	UserType = Type(2)
	// ChannelType represents an experiment on channel_id
	ChannelType = Type(3)
)

// ExperimentPlatform represents a platform for which to retrieve experiments
type ExperimentPlatform string

var (
	// LegacyPlatform is the default platform, for all existing experiments
	LegacyPlatform = ExperimentPlatform("experiments.json")
	// AbbyPlatform is the backend platform for AML team
	AbbyPlatform = ExperimentPlatform("abby-experiments.json")
	// TwilightPlatform is the platform for twilight-related experiments
	TwilightPlatform = ExperimentPlatform("twilight-experiments.json")
)

// Errors that are returned by this framework
var (
	ErrInvalidType = fmt.Errorf("experiment type incorrect")
	ErrNoDefault   = fmt.Errorf("all experiments must have a default")
	ErrOverRan     = fmt.Errorf("overran end of exp group list")
)

// Experiment contains the information on an experiment required to treat users into experimental groups.
// TODO: why is this exported
type Experiment struct {
	ID     string
	Name   string
	Salt   string
	Kind   Type
	Groups []Group
}

// Group represents an experimental group.
// TODO: why is this exported
type Group struct {
	Weight float64 `json:"weight"`
	Value  string  `json:"value"`
}

type experimentDefault struct {
	id           string
	kind         Type
	defaultGroup string
}

type experimentBranchTrackingEvent struct {
	Platform       string `json:"platform"`
	Group          string `json:"experiment_group"`
	ExperimentID   string `json:"experiment_id"`
	ExperimentType string `json:"experiment_type"`
	ExperimentName string `json:"experiment_name,omitempty"`
	UserID         string `json:"user_id,omitempty"`
	DeviceID       string `json:"device_id,omitempty"`
}
