package clipsv2

import (
	"fmt"
	"net/url"
	"time"
)

// GetTopClipsParams translates the request parameters to a URL query string.
type GetTopClipsParams struct {
	ChannelIDs *[]string
	GameNames  *[]string
	Languages  *[]string
	Start      *time.Time
	End        *time.Time
	Trending   bool
	Limit      int64
	Cursor     string

	// Note that WithURLTracking should only be true if the clip URLs can be retrieved from
	// the WithURLTracking API so that we can track the URL usage externally.
	WithURLTracking bool
}

// GetClipsByBroadcastIDParams translates the request parameters to a URL query string.
type GetClipsByBroadcastIDParams struct {
	BroadcastID string
	Sort        *string
	Limit       *int64
	Cursor      *string
}

// GetClipParams translates the request parameters to a URL query string
type GetClipParams struct {
	// Note that WithURLTracking should only be true if the clip URLs can be retrieved from
	// the WithURLTracking API so that we can track the URL usage externally.
	WithURLTracking bool
}

// NewGetTopClipsParams returns a GetTopClipsParams with default values.
// This should be used to initialize GetTopClipsParams.
func NewGetTopClipsParams() *GetTopClipsParams {
	return &GetTopClipsParams{
		Trending:        false,
		Limit:           100,
		WithURLTracking: false,
	}
}

// URLParamsString returns the url parameters string with the values in p.
func (p *GetTopClipsParams) URLParamsString() string {
	urlParams := fmt.Sprintf("with_url_tracking=%t&trending=%t&limit=%d", p.WithURLTracking, p.Trending, p.Limit)
	if p.Start != nil {
		urlParams += fmt.Sprintf("&start=%s", p.Start.Format(time.RFC3339))
	}

	if p.End != nil {
		urlParams += fmt.Sprintf("&end=%s", p.End.Format(time.RFC3339))
	}

	if p.Cursor != "" {
		urlParams += fmt.Sprintf("&cursor=%s", p.Cursor)
	}

	if p.ChannelIDs != nil {
		var channelIDParams string
		for _, channelID := range *p.ChannelIDs {
			channelIDParams += fmt.Sprintf("&channel_id=%s", url.QueryEscape(channelID))
		}
		urlParams += channelIDParams
	}

	if p.GameNames != nil {
		var gameParams string
		for _, game := range *p.GameNames {
			gameParams += fmt.Sprintf("&game=%s", url.QueryEscape(game))
		}
		urlParams += gameParams
	}

	if p.Languages != nil {
		var languagesParams string
		for _, language := range *p.Languages {
			languagesParams += fmt.Sprintf("&language=%s", url.QueryEscape(language))
		}
		urlParams += languagesParams
	}

	return urlParams
}

// NewGetClipParams returns a GetClipParams with default values.
// This should be used to initialize GetClipParams.
func NewGetClipParams() *GetClipParams {
	return &GetClipParams{
		WithURLTracking: false,
	}
}

// URLParamsString returns the url parameters string with the values in p.
func (p *GetClipParams) URLParamsString() string {
	return fmt.Sprintf("with_url_tracking=%t", p.WithURLTracking)
}
