package main

import (
	// core

	"fmt"
	"log"
	"net/url"
	"time"

	// app-specific

	badges "code.justin.tv/chat/badges/client"
	"code.justin.tv/video/clips-upload/adapters/db"
	"code.justin.tv/video/clips-upload/adapters/hystrixclients"
	"code.justin.tv/video/clips-upload/adapters/network"
	"code.justin.tv/video/clips-upload/adapters/repositories"
	"code.justin.tv/video/clips-upload/api"
	. "code.justin.tv/video/clips-upload/api/log"
	"code.justin.tv/video/clips-upload/app"
	"code.justin.tv/video/clips-upload/app/video"
	"code.justin.tv/video/clips-upload/clients"
	"code.justin.tv/video/clips-upload/clients/follows"
	"code.justin.tv/video/clips-upload/clients/kraken"
	"code.justin.tv/video/clips-upload/clients/mako"
	"code.justin.tv/video/clips-upload/clients/recommendations"
	"code.justin.tv/video/clips-upload/clients/spade"
	"code.justin.tv/video/clips-upload/datastore"
	i18n "code.justin.tv/video/clips-upload/i18n/client"
	"code.justin.tv/video/clips-upload/utils/lstats"
	"code.justin.tv/video/clips-upload/utils/slugs"
	"code.justin.tv/video/clips-upload/utils/stats"
	"code.justin.tv/vod/vodapi/rpc/vodapi"

	// vendor
	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/video/clips-upload/adapters/reporters"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/cactus/go-statsd-client/statsd"

	// video transcoding
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"

	appconfig "code.justin.tv/video/clips-upload/app/config"
)

// this should get over-written by using the -X linker flag
var serviceVersion = "development"

func throwUndefinedVariablePanic(varName string) {
	panic(fmt.Sprintf("environment variable %s not specified", varName))
}

func init() {
	config.Register(map[string]string{
		"ENVIRONMENT":                        "",
		"VIDEO_AWS_ACCESS_KEY":               "",
		"VIDEO_AWS_SECRET_KEY":               "",
		"ASSETS_ROOT_URL":                    "",
		"ASSETS_PROXY_URL":                   "",
		"TWITCH_API_URL":                     "",
		"TWITCH_API_INTERNAL_URL":            "",
		"USERS_SERVICE_URL":                  "",
		"MONEYPENNY_URL":                     "",
		"FEEDS_EDGE_URL":                     "",
		"FOLLOWING_SERVICE_URL":              "",
		"MAKO_URL":                           "",
		"CLIPS_ORIGIN_URL":                   "",
		"CLIPS_INFO_TABLE":                   "",
		"VIDEO_BUCKET":                       "",
		"RAW_MEDIA_BUCKET":                   "",
		"VIDEO_ROOT_URL":                     "",
		"RAW_VIDEO_ROOT_URL":                 "",
		"OWL_URL":                            "",
		"REDIS_URL":                          "",
		"SPADE_HOST_URL":                     "",
		"PG_DB_HOSTNAME":                     "",
		"PG_DB_PORT":                         "",
		"PG_DB_NAME":                         "",
		"PG_DB_USER":                         "",
		"PG_DB_PASS":                         "",
		"PG_SLAVE_HOSTNAME":                  "",
		"PG_SLAVE_PORT":                      "",
		"PG_SLAVE_NAME":                      "",
		"PG_SLAVE_USER":                      "",
		"PG_SLAVE_PASS":                      "",
		"RECOMMENDATIONS_URL":                "",
		"LEVIATHAN_AUTHORIZATION_TOKEN":      "",
		"LEVIATHAN_URL":                      "",
		"JAX_URL":                            "",
		"VIDEO_ROLE_ARN":                     "",
		"CLIPS_ROLE_ARN":                     "",
		"NOTIFICATIONS_SNS_TOPIC_ARN":        "",
		"MAKO_SNS_TOPIC_ARN":                 "",
		"TMI_HOSTPORT":                       "",
		"VODAPI_URL":                         "",
		"ZUMA_URL":                           "",
		"S3_ASSET_BUCKET":                    "",
		"S3_OUTPUT_PREFIX":                   "",
		"SNS_TOPIC_ARN":                      "",
		"REQUEST_QUEUE_URL":                  "",
		"ASSET_TRANSCODER_UPDATE_QUEUE":      "",
		"ASSET_TRANSCODER_S3_ASSET_BUCKET":   "",
		"ASSET_TRANSCODER_S3_OUTPUT_PREFIX":  "",
		"ASSET_TRANSCODER_SNS_TOPIC_ARN":     "",
		"ASSET_TRANSCODER_REQUEST_QUEUE_URL": "",
		"BADGES_HOST_URL":                    "",
		"FASTLY_SERVICE_ID":                  "",
		"FASTLY_TOKEN":                       "",
		"CLIP_MEDIA_EXPIRATION_BUCKET":       "",
	})
}

func main() {
	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	env := config.MustResolve("ENVIRONMENT")

	region := "us-west-2"
	awsAccessKey := config.MustResolve("VIDEO_AWS_ACCESS_KEY")
	awsSecretKey := config.MustResolve("VIDEO_AWS_SECRET_KEY")
	assetsRootURL := config.MustResolve("ASSETS_ROOT_URL")
	assetsProxyURL := config.Resolve("ASSETS_PROXY_URL")
	twitchAPIURL := config.MustResolve("TWITCH_API_URL")
	usersServiceURL := config.MustResolve("USERS_SERVICE_URL")
	moneypennyURL := config.MustResolve("MONEYPENNY_URL")
	feedsEdgeURL := config.MustResolve("FEEDS_EDGE_URL")
	recommendationsURL := config.MustResolve("RECOMMENDATIONS_URL")
	followingServiceURL := config.MustResolve("FOLLOWING_SERVICE_URL")
	makoURL := config.MustResolve("MAKO_URL")
	clipsOriginURL := config.MustResolve("CLIPS_ORIGIN_URL")
	videoBucket := config.MustResolve("VIDEO_BUCKET")
	rawClipMediaBucket := config.MustResolve("RAW_MEDIA_BUCKET")
	videoRootURL := config.MustResolve("VIDEO_ROOT_URL")
	rawVideoRootURL := config.MustResolve("RAW_VIDEO_ROOT_URL")
	owlURL := config.MustResolve("OWL_URL")
	redisURL := config.MustResolve("REDIS_URL")
	spadeHostURL := config.MustResolve("SPADE_HOST_URL")
	pgDBHost := config.MustResolve("PG_DB_HOSTNAME")
	pgDBPort := config.MustResolve("PG_DB_PORT")
	pgDBName := config.MustResolve("PG_DB_NAME")
	pgDBUser := config.MustResolve("PG_DB_USER")
	pgDBPass := config.MustResolve("PG_DB_PASS")
	pgSlaveHost := config.MustResolve("PG_SLAVE_HOSTNAME")
	pgSlavePort := config.MustResolve("PG_SLAVE_PORT")
	pgSlaveName := config.MustResolve("PG_SLAVE_NAME")
	pgSlaveUser := config.MustResolve("PG_SLAVE_USER")
	pgSlavePass := config.MustResolve("PG_SLAVE_PASS")
	leviathanAuthToken := config.MustResolve("LEVIATHAN_AUTHORIZATION_TOKEN")
	leviathanURL := config.MustResolve("LEVIATHAN_URL")
	jaxURL := config.MustResolve("JAX_URL")
	videoRoleARN := config.MustResolve("VIDEO_ROLE_ARN")
	clipsRoleARN := config.MustResolve("CLIPS_ROLE_ARN")
	notificiationsSNSTopicARN := config.MustResolve("NOTIFICATIONS_SNS_TOPIC_ARN")
	makoSNSTopicARN := config.MustResolve("MAKO_SNS_TOPIC_ARN")
	tmiHostPort := config.MustResolve("TMI_HOSTPORT")
	vodApiURL := config.MustResolve("VODAPI_URL")
	zumaURL := config.MustResolve("ZUMA_URL")
	assetTranscoderS3AssetBucket := config.MustResolve("ASSET_TRANSCODER_S3_ASSET_BUCKET")
	assetTranscoderS3OutputPrefix := config.MustResolve("ASSET_TRANSCODER_S3_OUTPUT_PREFIX")
	assetTranscoderSnsTopicArn := config.MustResolve("ASSET_TRANSCODER_SNS_TOPIC_ARN")
	assetTranscoderRequestQueueURL := config.MustResolve("ASSET_TRANSCODER_REQUEST_QUEUE_URL")
	assetTranscoderUpdateQueue := config.MustResolve("ASSET_TRANSCODER_UPDATE_QUEUE")
	badgesHostURL := config.MustResolve("BADGES_HOST_URL")
	fastlyServiceID := config.MustResolve("FASTLY_SERVICE_ID")
	fastlyToken := config.MustResolve("FASTLY_TOKEN")
	clipMediaExpirationBucket := config.MustResolve("CLIP_MEDIA_EXPIRATION_BUCKET")

	appConfig := appconfig.Config{
		NotificationSNSTopicARN:       notificiationsSNSTopicARN,
		MakoSNSTopicARN:               makoSNSTopicARN,
		AssetTranscoderS3AssetBucket:  assetTranscoderS3AssetBucket,
		AssetTranscoderS3OutputPrefix: assetTranscoderS3OutputPrefix,
		AssetTranscoderSNSTopicARN:    assetTranscoderSnsTopicArn,
		RequestQueueUrl:               assetTranscoderRequestQueueURL,
		AssetTranscoderUpdateQueue:    assetTranscoderUpdateQueue,
		VideoBucket:                   videoBucket,
		RawVideoBucket:                rawClipMediaBucket,
		RawVideoRootURL:               rawVideoRootURL,
	}

	assetsURLWithServiceVerison := fmt.Sprintf("%s/%s", assetsRootURL, serviceVersion)

	if env == "development" && assetsProxyURL != "" && assetsRootURL != assetsProxyURL {
		assetsRootURL = assetsProxyURL
		assetsURLWithServiceVerison = assetsProxyURL
	}

	_, err = url.Parse(rawVideoRootURL)
	if err != nil {
		panic("raw video root URL could not be parsed")
	}

	logger := NewMultiLogger(NewLogrusLogger(), NewRollbarLogger())

	statsClient, err := stats.NewStats(env)
	if err != nil {
		log.Fatal(err)
	}

	logStats, err := lstats.NewClient(statsClient, logger)
	if err != nil {
		logger.Error(err)
		return
	}

	defer func() {
		err := statsClient.Close()
		if err != nil {
			log.Print(err.Error())
		}
	}()

	spadeClient, err := spade.NewClient(spadeHostURL, logStats, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create spade event reporter %s", err.Error()))
		return
	}

	twitchAPIHTTPClient, err := network.NewNetworkHttpClient(twitchAPIURL, "twitch_api", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	moneypennyClient, err := clients.NewMoneypennyClient(moneypennyURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create moneypenny service client %s", err.Error()))
		return
	}

	feedsClient, err := clients.NewFeedsClient(feedsEdgeURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create feeds client %s", err.Error()))
	}

	recommendationsClient, err := recommendations.NewClient(recommendationsURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create recommendations client %s", err.Error()))
	}

	followsClient := follows.NewClient(followingServiceURL, statsClient)

	makoClient := mako.NewClient(makoURL, statsClient)

	usersServiceClient, err := clients.NewUsersServiceClient(usersServiceURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create users service client %s", err.Error()))
		return
	}

	channelsClient, err := clients.NewChannelsClient(usersServiceURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create channels client %s", err.Error()))
		return
	}

	userPropertiesHTTPClient, err := network.NewNetworkHttpClient("https://api.internal.twitch.tv", "user_properties", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	usherHTTPClient, err := network.NewNetworkHttpClient("filler", "usher", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	leviathanHTTPClient, err := network.NewNetworkHttpClient(leviathanURL, "leviathan", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	clipCreatorHTTPClient, err := network.NewNetworkHttpClient(assetsRootURL, "clip_creator", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	fastlyHttpClient, err := network.NewNetworkHttpClient("https://api.fastly.com", "fastly", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	jaxClient, err := clients.NewJaxClient(jaxURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	zumaClient, err := clients.NewZumaClient(zumaURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create zuma client %s", err.Error()))
		return
	}

	krakenClient, err := kraken.NewClient(twitchAPIURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create kraken client %s", err.Error()))
		return
	}

	videoS3Client := configureS3Client(videoRoleARN)
	clipsS3Client := configureS3Client(clipsRoleARN)
	videoSQSClient := configureSQSClient(videoRoleARN)
	clipsSQSClient := configureSQSClient(clipsRoleARN)

	dataRepository := repositories.NewS3DataRepository(
		region,
		videoBucket,
		rawClipMediaBucket,
		clipMediaExpirationBucket,
		s3.BucketCannedACLPublicRead,
		videoS3Client,
		clipsS3Client)

	channelRepository := repositories.NewTwitchChannelRepository(twitchAPIHTTPClient, statsClient, logger)
	vodRepository := repositories.NewTwitchVODRepository(twitchAPIHTTPClient, statsClient, logger)

	owlClient, err := clients.NewOwlClient(owlURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create owl client from owl/client library %s", err.Error()))
		return
	}
	hystrixclients.InitHystrix(logger)
	owlHystrixClient := hystrixclients.NewOwlClient(owlClient)

	oauthTokenRepository := repositories.NewOauthTokenRepository(twitchAPIHTTPClient, twitchAPIURL, owlHystrixClient, statsClient, logger)
	userPropertiesRepository := repositories.NewUserPropertiesRepository(userPropertiesHTTPClient, logger)

	usherClient := clients.NewUsherClient(usherHTTPClient, video.HLSPlaylistParser, logStats, logger)

	redisClient, err := db.NewRedisClient(redisURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to redis client: %s", err.Error()))
		return
	}

	pgConn, err := db.NewPostgresConn(pgDBHost, pgDBPort, pgDBName, pgDBUser, pgDBPass, logStats)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create connection to database: %s", err.Error()))
		return
	}
	go db.CollectMetrics(pgConn, statsClient)

	pgSlaveConn, err := db.NewPostgresConn(pgSlaveHost, pgSlavePort, pgSlaveName, pgSlaveUser, pgSlavePass, logStats)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create connection to slave: %s", err.Error()))
		return
	}
	go db.CollectMetrics(pgSlaveConn, statsClient)

	clipsDatastore := datastore.NewDatastore(pgConn, pgSlaveConn, redisClient, logStats, logger)

	viewCountsRepository := repositories.NewViewCountsRepository(pgConn, redisClient, statsClient, logger)
	acknowledgementRepository := repositories.NewAcknowledgementRepository(pgConn, statsClient, logger)

	slugGenerator := slugs.NewGenerator(logStats)

	cmdRunner := video.NewCommandRunner(logStats, logger)

	languageMatcher := i18n.GetSupportedLanguageMatcher()
	clientI18N := i18n.NewClientI18N(languageMatcher)

	vodApiConf := twitchclient.ClientConf{
		Host: vodApiURL,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 200,
		},
		Stats: statsClient,
	}
	vodApiClient := vodapi.NewVodApiProtobufClient(vodApiURL, twitchclient.NewHTTPClient(vodApiConf))

	tmiClient, err := clients.NewTMIClient(tmiHostPort, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create tmi client %s", err))
		return
	}

	snsClient := configureSNSClient(videoRoleARN)

	badgesClient, err := createBadgesClient(badgesHostURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create badges client %s", err))
		return
	}

	clients := &clients.Clients{
		I18N:                 clientI18N,
		OauthTokenRepository: oauthTokenRepository,
		UsersServiceClient:   usersServiceClient,
		ChannelsClient:       channelsClient,
		VodApiClient:         vodApiClient,
		ZumaClient:           zumaClient,
		MoneypennyClient:     moneypennyClient,
		HttpClient:           clipCreatorHTTPClient,
		VideoSQSClient:       videoSQSClient,
		ClipsSQSClient:       clipsSQSClient,
		VideoS3Client:        videoS3Client,
		RedisClient:          redisClient,
		ViewCountsRepo:       viewCountsRepository,
		SNSClient:            snsClient,
		KrakenClient:         krakenClient,
		UsherClient:          usherClient,
		DataRepo:             dataRepository,
		BadgesClient:         badgesClient,
	}

	clipCreator := video.NewClipCreator(
		dataRepository,
		clipsDatastore,
		cmdRunner,
		logStats,
		logger,
		rawVideoRootURL,
		clipCreatorHTTPClient,
		appConfig,
		clients,
		true,
	)

	clipEditor := video.NewClipEditor(cmdRunner, clipCreatorHTTPClient)
	transcoder := video.NewTranscoder(cmdRunner)

	leviathanClipReporter := reporters.NewLeviathanClipReporter(leviathanHTTPClient, leviathanAuthToken, leviathanURL, logger)
	cacheInvalidator := network.NewCDNCacheInvalidator(region, awsAccessKey, awsSecretKey, fastlyServiceID, fastlyToken, fastlyHttpClient)

	app := app.NewApp(
		clipsDatastore,
		clients,
		dataRepository,
		slugGenerator,
		channelRepository,
		userPropertiesRepository,
		feedsClient,
		jaxClient,
		recommendationsClient,
		followsClient,
		makoClient,
		clipCreator,
		leviathanClipReporter,
		vodRepository,
		acknowledgementRepository,
		oauthTokenRepository,
		cacheInvalidator,
		spadeClient,
		videoRootURL,
		redisClient,
		tmiClient,
		appConfig,
		logStats,
		logger,
		cmdRunner,
		clipEditor,
		transcoder,
	)

	// server
	server := api.NewServer(
		clients,
		redisClient,
		spadeClient,
		assetsURLWithServiceVerison,
		clipsOriginURL,
		twitchAPIURL,
		videoRootURL,
		rawVideoRootURL,
		serviceVersion,
		logStats,
		logger,
		app,
	)

	// By default this handles SIGUSR2, SIGINT, SIGTERM OS signals.
	// https://git-aws.internal.justin.tv/common/twitchhttp/blob/282c1a72b779c0cf35c3a9d406cb02af03a29e18/server.go#L110
	twitchhttp.AddDefaultSignalHandlers()
	serverConfig := twitchhttp.NewConfig()
	// 30 seconds is the longest it takes for a clip to be created / published.
	serverConfig.GracefulTimeout = 30 * time.Second
	serverConfig.Statter = statsClient
	log.Fatal(twitchhttp.ListenAndServe(server, serverConfig))
}

func configureSNSClient(role string) *sns.SNS {
	stsclient := sts.New(session.New(&aws.Config{
		Region: aws.String("us-west-2"),
	}))

	arp := &stscreds.AssumeRoleProvider{
		ExpiryWindow: 1 * time.Second,
		RoleARN:      role,
		Client:       stsclient,
	}

	creds := credentials.NewCredentials(arp)

	session := session.New(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	return sns.New(session)
}

func configureS3Client(role string) *s3.S3 {
	stsclient := sts.New(session.New(&aws.Config{
		Region: aws.String("us-west-2"),
	}))

	arp := &stscreds.AssumeRoleProvider{
		ExpiryWindow: 1 * time.Second,
		RoleARN:      role,
		Client:       stsclient,
	}

	creds := credentials.NewCredentials(arp)

	session := session.New(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	return s3.New(session)
}

func configureSQSClient(role string) *sqs.SQS {
	stsclient := sts.New(session.New(&aws.Config{
		Region: aws.String("us-west-2"),
	}))

	arp := &stscreds.AssumeRoleProvider{
		ExpiryWindow: 1 * time.Second,
		RoleARN:      role,
		Client:       stsclient,
	}

	creds := credentials.NewCredentials(arp)

	session := session.New(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	return sqs.New(session)
}

func createBadgesClient(host string, stats statsd.Statter) (badges.Client, error) {
	transport := twitchclient.TransportConf{
		MaxIdleConnsPerHost: 50,
	}

	conf := twitchclient.ClientConf{
		Host:      host,
		Transport: transport,
		Stats:     stats,
	}
	return badges.NewClient(conf)
}
