package main

import (
	"context"
	"fmt"
	"log"
	"net/http"
	"net/url"
	"os"
	"os/signal"
	"strings"
	"sync"
	"syscall"
	"time"

	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	validator "gopkg.in/validator.v2"

	"github.com/abh/geoip"
	_ "github.com/lib/pq"
	"github.com/sirupsen/logrus"

	"code.justin.tv/chat/db"
	"code.justin.tv/feeds/following-service/client/follows"
	evs "code.justin.tv/growth/emailvalidator/evs/client"
	"code.justin.tv/web/users-service/internal/clients/payments"
	"code.justin.tv/web/users-service/internal/clients/usher"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/foundation/xray"
	rediserclient "code.justin.tv/identity/rediser/client"
	rediser "code.justin.tv/identity/rediser/common"
	owl "code.justin.tv/web/owl/client"
	"code.justin.tv/web/users-service/api"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/channels"
	"code.justin.tv/web/users-service/backend/reservations"
	"code.justin.tv/web/users-service/backend/sdk"
	"code.justin.tv/web/users-service/backend/users"
	cacher "code.justin.tv/web/users-service/backend/users/cache"
	"code.justin.tv/web/users-service/backend/users/cacheddb"
	udb "code.justin.tv/web/users-service/backend/users/db"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/internal/auth"
	"code.justin.tv/web/users-service/internal/clients/auditor"
	"code.justin.tv/web/users-service/internal/clients/cache/cacherwrapper"
	"code.justin.tv/web/users-service/internal/clients/cache/rediscacher"
	"code.justin.tv/web/users-service/internal/clients/discovery"
	"code.justin.tv/web/users-service/internal/clients/gateway"
	"code.justin.tv/web/users-service/internal/clients/hystrix"
	"code.justin.tv/web/users-service/internal/clients/ipblock"
	"code.justin.tv/web/users-service/internal/clients/kinesis"
	"code.justin.tv/web/users-service/internal/clients/partnerships"
	"code.justin.tv/web/users-service/internal/clients/s3"
	"code.justin.tv/web/users-service/internal/clients/sns"
	"code.justin.tv/web/users-service/internal/clients/twilio"
	"code.justin.tv/web/users-service/internal/clients/zuma"

	"code.justin.tv/web/users-service/internal/clients/uploader"

	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/internal/worker/emailvalidationsuccess"
	"code.justin.tv/web/users-service/internal/worker/expirecachequeue"
	"code.justin.tv/web/users-service/internal/worker/imageuploadqueue"

	"code.justin.tv/web/users-service/internal/clients/pubsub"
	"code.justin.tv/web/users-service/internal/image"
	"code.justin.tv/web/users-service/internal/worker/deadletterqueue"
	"code.justin.tv/web/users-service/logic"
)

const (
	workerWaitTimeout        = 20 * time.Second
	jobWaitTimeout           = 10 * time.Second
	rediserMetricsSampleRate = 0.1
)

var (
	ctx = context.Background()
	// Keeping sandstorm configuration separate for beanstalk and courier to prevent accidentally affecting courier environments
	beanstalkSandstormConfigs = []configs.SandstormConfig{
		{
			Environment: "staging",
			Region:      "us-west-2",
			Role:        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-users-service-staging-us-west-2",
		}, {
			Environment: "production",
			Region:      "us-west-2",
			Role:        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-users-service-production-us-west-2",
		}, {
			Environment: "staging",
			Region:      "us-east-1",
			Role:        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-users-service-staging-us-east-1",
		},
		{
			Environment: "production",
			Region:      "us-east-1",
			Role:        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-users-service-production-us-east-1",
		},
	}
	courierSandstormConfigs = []configs.SandstormConfig{
		{
			Environment: "staging",
			Region:      "us-west-2",
			Role:        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-users-service-staging",
		}, {
			Environment: "production",
			Region:      "us-west-2",
			Role:        "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-users-service-production",
		},
	}
)

func main() {
	environment := configs.Environment()
	region := configs.Region()
	infra := os.Getenv("INFRA")

	// start up server by default
	startup := "server"
	if len(os.Args) > 1 {
		startup = os.Args[1]
	}

	// Add startup type to all logx (rollbar) errors
	ctx = logx.WithFields(ctx, logx.Fields{
		"region":  region,
		"startup": startup,
	})

	conf := configs.Defined.Value(environment, region)

	sandstormConfigs := courierSandstormConfigs
	if infra == "beanstalk" {
		sandstormConfigs = beanstalkSandstormConfigs
	}

	sandstorm, err := configs.NewSandstorm(environment, region, sandstormConfigs)
	if err == nil {
		secretFiles, err := configs.LoadSecrets(&conf, sandstorm)
		if err != nil {
			log.Fatal("failed to configure secrets:", err)
		}

		defer func() {
			if err := secretFiles.Close(); err != nil {
				logx.Error(ctx, "failed to close secret files", logx.Fields{
					"Error": fmt.Sprintf("%v", err),
				})
			}
		}()
	} else if err != nil && err != configs.ErrNoSandstormConfig {
		log.Fatalf("failed to configure sandstorm for %q/%q: %v", environment, region, err)
	}

	// The assume role value should only be nonzero in our new account on beanstalk
	if infra != "beanstalk" && region != "us-west-2" {
		conf.STSAssumeRole = ""
	}

	creds := defaultCredentials(conf.STSAssumeRole)

	logx.InitDefaultLogger(logx.Config{
		RollbarToken: conf.RollbarToken,
		RollbarEnv:   environment,
		RollbarLevels: []logrus.Level{
			logrus.PanicLevel,
			logrus.FatalLevel,
			logrus.ErrorLevel,
			logrus.WarnLevel,
		},
	})
	defer logx.Wait()

	if err := config.SetStatsClients(config.StatsConf{
		HostPort:    conf.Stats.HostPort,
		App:         conf.Stats.App,
		AWSRegion:   region,
		Environment: configs.EnvironmentRegion(),
	}); err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure config stats clients:", err))
	}

	// The client configured here is referenced by packages as config.RollbarErrorLogger()
	if err := config.SetRollbarClient(config.RollbarConf{
		Token:       conf.RollbarToken,
		Environment: environment,
	}); err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure config rollbar client:", err))
	}

	sns, err := configureSNSPublisher(conf.Topics, creds)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure sns publisher:", err))
		return
	}

	userCacheBackend, err := configureCacheBackend("user-props-string", environment, sns, conf)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure user cache backend:", err))
	}

	defer func() {
		if err := userCacheBackend.Close(); err != nil {
			logx.Error(ctx, err)
		}
	}()

	s3, err := configureS3(conf.LegacyAWSKey, conf.LegacyAWSSecret)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to create S3 client", err))
	}

	ub, err := configureUserBackend(userCacheBackend, s3, conf.SiteDB)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure user backend:", err))
	}

	ushr, err := configureUsherClient(conf.Hosts.Usher)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure usher client:", err))
	}

	paym, err := configurePaymentsClient(conf.Hosts.Payments)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure payments client:", err))
	}

	evs, err := configureEVSClient(conf.Hosts.EVS)
	if err != nil {
		log.Fatal("failed to configure email validation client:", err)
		return
	}

	follows, err := configureFollowsClient(conf.Hosts.Follows)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure followers client:", err))
		return
	}

	ib := configureIPBlockClient(conf.Hosts.IPBlock)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure ipblock client:", err))
	}

	gw := configureGatewayClient(conf.Hosts.Gateway)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure gateway client:", err))
	}

	kc, err := configureKinesisPublisher(conf.UserMutations)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure kinesis publisher:", err))
		return
	}

	partners := configurePartnershipsClient(conf.Hosts.Partnerships)

	spade, err := configureSpadeClient(conf.Hosts.Spade)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure spade client:", err))
		return
	}

	err = configureXRay(conf.XraySampling)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure xray:", err))
		return
	}

	channelCacheBackend, err := configureCacheBackend("channel-props-string", environment, sns, conf)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure channel cache backend:", err))
	}
	defer func() {
		if err := channelCacheBackend.Close(); err != nil {
			logx.Error(ctx, err)
		}
	}()

	channels, err := configureChannelsBackend(channelCacheBackend, conf.SiteDB)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure channels backend:", err))
		return
	}

	auditor, err := configureAuditor(conf.Hosts.Auditor)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure auditor:", err))
		return
	}

	d, err := configureDiscovery(conf.Hosts.Discovery)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure dicovery:", err))
		return
	}

	var twi twilio.Client
	if !conf.Twilio.Disabled {
		twi, err = configureTwilio(conf.Twilio)
		if err != nil {
			logx.Fatal(ctx, fmt.Sprint("failed to configure twilio client:", err))
			return
		}
	}

	geoIP, err := configureGeoIP(conf.GeoIPPath)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure geoIP database:", err))
		return
	}

	uploader, err := configureUploader(conf.Hosts.UploadService, conf.Topics.ImageUploadEvents)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure image uploader:", err))
		return
	}

	reservations, err := configureReservationsBackend(conf.Reservations)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure reservations backend:", err))
		return
	}

	sdk, err := configureSDKBackend(conf.SiteDB)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure sdk backend:", err))
		return
	}

	owlClient, err := configureOwl(conf.Hosts.Owl)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure owl client:", err))
	}

	zumaClient, err := configureZuma(conf.Hosts.Zuma)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("Failed to configure zuma client:", err))
	}

	pubclient, err := pubsub.NewPubSub(conf.Hosts.PubSub)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to create pubsub client", err))
	}

	hystrix.InitHystrix(conf.HystrixSampling, conf.HystrixErrorSampling)

	l, err := logic.New(ub, follows, ib, gw, kc, partners, spade, sns, channels, auditor, twi, d, geoIP, uploader, evs, reservations, owlClient, zumaClient, s3, conf.SeedVerifyCode, pubclient, sdk, ushr, paym)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to create logic", err))
	}

	decoder, err := auth.NewDecoder(conf.CartmanKeyPath)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to create cartman decoder", err))
	}

	localExpirationCacher, err := configureLocalExpirationCacher(environment, conf)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure local expiration cacher", err))
	}

	logx.Info(ctx, "sqs region is "+conf.SQSRegion)
	workerConfs := setWorkerConfigs(l, uploader, s3, pubclient, localExpirationCacher, conf)

	err = image.SetUpImageReader()
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to set up image reader", err))
	}

	switch startup {
	case "worker":
		stop := make(chan struct{})

		tsConf := twitchserver.NewConfig()
		tsConf.Stop = stop
		go func() {
			if err := twitchserver.ListenAndServe(twitchserver.NewServer(), tsConf); err != nil {
				logx.Fatal(ctx, fmt.Sprint("error occurred during listen and serving:", err))
			}
		}()

		workerClients, err := worker.InitClients(workerConfs)
		if err != nil {
			logx.Fatal(ctx, fmt.Sprint("failed to configure worker clients:", err))
			return
		}

		if err := setupAndInitEventWorkers(ctx, conf.Workers, workerConfs, workerClients); err != nil {
			logx.Fatal(ctx, err)
		}

		close(stop)
	case "server":
		server, err := api.NewServer(l, decoder)
		if err != nil {
			logx.Fatal(ctx, fmt.Sprint("failed to create new server:", err))
		}

		tsConf := twitchserver.NewConfig()
		if err := twitchserver.ListenAndServe(server, tsConf); err != nil {
			logx.Fatal(ctx, fmt.Sprint("error occurred during listen and serving:", err))
		}
	default:
		logx.Fatal(ctx, "unknown startup type", logx.Fields{
			"startup": startup,
		})
	}
}

func defaultCredentials(roleArn string) *credentials.Credentials {
	sess := session.New()

	return credentials.NewChainCredentials([]credentials.Provider{
		&stscreds.AssumeRoleProvider{
			RoleARN: roleArn,
			Client:  sts.New(sess),
		},
		&ec2rolecreds.EC2RoleProvider{
			Client: ec2metadata.New(sess),
		},
		&credentials.EnvProvider{},
		&credentials.SharedCredentialsProvider{},
	})
}

func configureFollowsClient(host string) (follows.Client, error) {
	return follows.NewClient(twitchhttp.ClientConf{
		Host:  host,
		Stats: config.Statsd(),
	})
}

func configurePartnershipsClient(host string) partnerships.Client {
	return partnerships.NewRipleyClient(twitchclient.ClientConf{
		Host:  host,
		Stats: config.Statsd(),
	})
}

func configureXRay(sample float64) error {
	return xray.Configure(xray.Config{
		Name:     "users-service",
		Sampling: sample,
	})
}

func configureSpadeClient(host string) (spade.Client, error) {
	httpClient := &http.Client{}
	return spade.NewClient(
		spade.InitHTTPClient(httpClient),
		spade.InitMaxConcurrency(1000),
		spade.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			err := config.Statsd().Timing(fmt.Sprintf("spade.%s.%d", name, httpStatus), int64(dur), 1.0)
			if err != nil {
				log.Printf("Error emitting spade stat %v\n", err)
			}
		}),
		spade.InitBaseURL(url.URL{
			Scheme: "https",
			Host:   host,
		}),
	)
}

func configureEVSClient(host string) (evs.Client, error) {
	return evs.NewClient(twitchclient.ClientConf{
		Host:  host,
		Stats: config.Statsd(),
	})
}

func configureUsherClient(host string) (usher.Client, error) {
	return usher.NewClient(host, config.Statsd())
}

func configurePaymentsClient(host string) (payments.Client, error) {
	return payments.NewClient(host, config.Statsd())
}

func configureIPBlockClient(host string) ipblock.Client {
	return ipblock.New(host)
}

func configureGatewayClient(host string) gateway.Client {
	return gateway.New(host)
}

func configureKinesisPublisher(conf configs.KinesisStream) (kinesis.Publisher, error) {
	return kinesis.NewPublisher(
		conf.Region,
		conf.RetryCount,
		conf.RetryDelay,
		conf.Role,
		conf.Name,
		config.Statsd())
}

func configureSNSPublisher(topics configs.Topics, creds *credentials.Credentials) (sns.Publisher, error) {
	conf := sns.Config{
		Regions:                 strings.Split(topics.SNSRegions, ","),
		UserModerationEventsARN: topics.ModerationEvents,
		UserRenameEventsARN:     topics.RenameEvents,
		UserCreationEventARN:    topics.CreationEvents,
		UserMutationEventARN:    topics.MutationEvents,
		ChannelMutationEventARN: topics.ChannelMutationEvents,
		PushyDispatchEventARN:   topics.PushyDispatched,
		UserSoftDeleteEventsARN: topics.UserSoftDeleteEvents,
		UserHardDeleteEventsARN: topics.UserHardDeleteEvents,
		UserUndeleteEventsARN:   topics.UserUndeleteEvents,
		ExpireCacheEventsARN:    topics.ExpireCacheEvents,
		Credentials:             creds,
	}
	return sns.NewPublisher(conf, config.Statsd())
}

func configureS3(key, secret string) (s3.S3Client, error) {
	o := s3.OldCredentialProvider{
		Key:    key,
		Secret: secret,
	}
	return s3.New(o)
}

func configureUserBackend(backendCacher backend.Cacher, s3 s3.S3Client, dbConf configs.DB) (users.Backend, error) {
	sdb, err := configureDb(dbConf, dbConf.Slave)
	if err != nil {
		return nil, err
	}

	mdb, err := configureDb(dbConf, dbConf.Master)
	if err != nil {
		return nil, err
	}

	sdbQuerier, err := configureDbQuerier(sdb)
	if err != nil {
		return nil, err
	}

	mdbQuerier, err := configureDbQuerier(mdb)
	if err != nil {
		return nil, err
	}

	hystrixB := &users.HystrixBackend{Backend: udb.New(mdbQuerier, sdbQuerier, s3)}

	ub, err := cacheddb.NewBackend(hystrixB, cacher.New(backendCacher))
	if err != nil {
		return nil, err
	}

	return ub, nil
}

func configureChannelsBackend(backendCacher backend.Cacher, dbConf configs.DB) (channels.Backend, error) {
	sdb, err := configureDb(dbConf, dbConf.Slave)
	if err != nil {
		return nil, err
	}

	mdb, err := configureDb(dbConf, dbConf.Master)
	if err != nil {
		return nil, err
	}

	sdbQuerier, err := configureDbQuerier(sdb)
	if err != nil {
		return nil, err
	}

	mdbQuerier, err := configureDbQuerier(mdb)
	if err != nil {
		return nil, err
	}

	b, err := channels.NewBackend(sdbQuerier, mdbQuerier)
	if err != nil {
		return nil, err
	}

	hystrixB := &channels.HystrixBackend{Backend: b}

	return channels.NewCachedBackend(hystrixB, backendCacher)
}

func configureReservationsBackend(dbConf configs.DB) (reservations.Backend, error) {
	sdb, err := configureDb(dbConf, dbConf.Slave)
	if err != nil {
		return nil, err
	}

	mdb, err := configureDb(dbConf, dbConf.Master)
	if err != nil {
		return nil, err
	}

	sdbQuerier, err := configureDbQuerier(sdb)
	if err != nil {
		return nil, err
	}

	mdbQuerier, err := configureDbQuerier(mdb)
	if err != nil {
		return nil, err
	}

	backend, err := reservations.NewBackend(sdbQuerier, mdbQuerier)

	return &reservations.HystrixReservationsBackend{Backend: backend}, nil
}

func configureSDKBackend(dbConf configs.DB) (sdk.Backend, error) {
	mdb, err := configureDb(dbConf, dbConf.Master)
	if err != nil {
		return nil, err
	}
	mdbQuerier, err := configureDbQuerier(mdb)
	if err != nil {
		return nil, err
	}
	backend, err := sdk.NewBackend(mdbQuerier)
	return &sdk.HystrixBackend{Backend: backend}, nil
}

func configureRediser(cacheType string, env string, conf configs.Redis) (rediserclient.Handler, error) {
	opts := &rediser.Options{
		Addrs:           []string{conf.Host},
		KeyPrefix:       fmt.Sprintf("users:%v", env),
		StatPrefix:      cacheType,
		StatSampleRate:  rediserMetricsSampleRate,
		MonitorInterval: 5 * time.Second,
		PoolSize:        conf.MaxConnections,
		DialTimeout:     time.Duration(conf.ConnectTimeout) * time.Millisecond,
		ReadTimeout:     time.Duration(conf.ReadTimeout) * time.Millisecond,
		WriteTimeout:    time.Duration(conf.WriteTimeout) * time.Millisecond,
		UseReadReplicas: true,
	}
	client, err := rediserclient.NewClient(opts, config.Statsd())
	if err != nil {
		return nil, fmt.Errorf("Could not initialize rediser. err=%v", err)
	}
	return client, nil
}
func configureDb(dbConf configs.DB, hostConf configs.DBHost) (db.DB, error) {
	password, err := dbConf.ResolvePassword()
	if err != nil {
		return nil, err
	}

	log.Printf("Connecting to %v:%d as %v using DB %v",
		hostConf.Host,
		hostConf.Port,
		dbConf.User,
		hostConf.Name)

	db, err := db.Open(
		db.DriverName("postgres"),
		db.Host(hostConf.Host),
		db.Port(hostConf.Port),
		db.User(dbConf.User),
		db.Password(password),
		db.DBName(hostConf.Name),
		db.MaxOpenConns(hostConf.MaxOpenConns),
		db.MaxIdleConns(hostConf.MaxIdleConns),
		db.MaxQueueSize(hostConf.MaxQueueSize),
		db.ConnAcquireTimeout(hostConf.ConnAcquireTimeout),
		db.RequestTimeout(hostConf.RequestTimeout),
		db.MaxConnAge(hostConf.MaxConnAge),
	)
	if err != nil {
		return nil, err
	}

	logger, err := database.NewLogger(config.Statsd(), hostConf.LoggerPrefix)
	db.SetCallbacks(logger.LogDBStat, logger.LogRunStat)

	go func() {
		ticker := time.Tick(10 * time.Second)
		for {
			select {
			case <-ticker:
				logger.LogDBState(db.Info())
			}
		}
	}()

	return db, nil
}

func configureDbQuerier(db db.DB) (database.Querier, error) {
	return database.NewQuerier(db)
}

func configureLocalExpirationCacher(env string, conf configs.Config) (rediscacher.LocalExpirationCacher, error) {
	// Configure primary
	primary, err := configureRedisCacher("redis_primary", env, conf.Cache.Primary)
	if err != nil {
		return nil, err
	}

	// Configure Backup
	backup, err := configureRedisCacher("redis_backup", env, conf.Cache.Backup)
	if err != nil {
		return nil, err
	}

	return rediscacher.NewLocalExpirationCacher(primary, backup), nil
}

func configureCacheBackend(key string, env string, sns sns.Publisher, conf configs.Config) (backend.Cacher, error) {
	var cachebackend backend.Cacher

	// Configure primary
	primary, err := configureRedisCacher("redis_primary", env, conf.Cache.Primary)
	if err != nil {
		return nil, err
	}

	// Configure Backup
	backup, err := configureRedisCacher("redis_backup", env, conf.Cache.Backup)
	if err != nil {
		return nil, err
	}

	cachebackend = rediscacher.NewMarshaler(primary, backup, time.Duration(conf.RedisTTL)*time.Millisecond, key+"-gz", &rediscacher.GZipMarshaler{Marshaler: &rediscacher.JSONMarshaler{}}, sns, &rediscacher.Flags{})
	cacherwrapper := cacherwrapper.New(cachebackend, key, "redis")
	return cacherwrapper, nil
}

func configureRedisCacher(cacheType string, env string, conf configs.Redis) (rediscacher.Redis, error) {
	cache, err := configureRediser(cacheType, env, conf)
	if err != nil {
		return nil, err
	}
	return cacherwrapper.NewRediser(cache), nil
}

func configureAuditor(host string) (auditor.Auditor, error) {
	conf := twitchhttp.ClientConf{
		Host: host,
	}
	return auditor.NewAuditor(conf)
}

func configureTwilio(conf configs.Twilio) (twilio.Client, error) {
	return twilio.NewClient(twilio.Config{
		AccountSSID: conf.Account,
		AuthToken:   conf.Auth,
		FromChoices: strings.Split(conf.PhoneNumbers, ","),
	})
}

func configureGeoIP(path string) (*geoip.GeoIP, error) {
	return geoip.Open(path)
}

func configureDiscovery(host string) (discovery.Discovery, error) {
	conf := twitchhttp.ClientConf{
		Host: host,
	}
	return discovery.NewDiscovery(conf)
}

func configureUploader(host, arn string) (uploader.Uploader, error) {
	return uploader.NewUploader(host, arn)
}

func configureOwl(host string) (owl.Client, error) {
	return owl.NewClient(twitchhttp.ClientConf{
		Host: host,
	})
}

func configureZuma(host string) (zuma.Zuma, error) {
	return zuma.NewZuma(twitchclient.ClientConf{
		Host: host,
	})
}

func setWorkerConfigs(logic logic.Logic, uploader uploader.Uploader, s3 s3.S3Client, pubclient pubsub.PubSub, lec rediscacher.LocalExpirationCacher, conf configs.Config) worker.Configs {
	return worker.Configs{
		SQSRegion:             conf.SQSRegion,
		SQSEndpoint:           conf.SQSEndpoint,
		Stats:                 config.Statsd(),
		Users:                 logic,
		Uploader:              uploader,
		S3:                    s3,
		PubSub:                pubclient,
		Topics:                map[string]string{worker.EmailValidationQueue: conf.Topics.EmailVerified, worker.ImageUploadQueue: conf.Topics.ImageUploadEvents},
		LocalExpirationCacher: lec,
	}
}

func setupAndInitEventWorkers(ctx context.Context, queues configs.WorkerQueues, conf worker.Configs, clients worker.Clients) error {
	done := make(chan struct{})

	sendDone := func(wgs []*sync.WaitGroup) {
		if len(wgs) == 0 {
			return
		}

		close(done)
	}

	sig := make(chan os.Signal, 1)
	signal.Notify(sig, syscall.SIGTERM, syscall.SIGUSR2)

	wgs, err := initEventWorkers(ctx, queues, conf, clients, done)
	defer func() {
		log.Printf("waiting for workers to complete")
		for i, wg := range wgs {
			wg.Wait()
			log.Printf("worker %d shutdown", i)
		}
		log.Printf("workers completed")
	}()
	if err != nil {
		log.Printf("worker failed, shutting down")
		sendDone(wgs)
		return err
	}

	<-sig
	sendDone(wgs)
	return nil
}

// initEventWorkers spins up workers listening to SQS queues.
// Introducing new push notifications requires intializing workers here.
func initEventWorkers(ctx context.Context, queues configs.WorkerQueues, conf worker.Configs, clients worker.Clients, done <-chan struct{}) ([]*sync.WaitGroup, error) {
	wgs := []*sync.WaitGroup{}

	log.Printf("Workers initialized: \n EmailValidatorWorker: %v, ImageUploadWorker: %v, DeadLetterWorker: %v, ExpireCacheWorker: %v",
		!queues.EmailVerified.Disabled,
		!queues.ImageUpload.Disabled,
		!queues.DeadLetter.Disabled,
		!queues.ExpireCache.Disabled)

	createWorker := func(conf configs.WorkerQueue, handleFn worker.EventHandler) (*sync.WaitGroup, error) {
		if conf.Disabled {
			return nil, nil
		}

		params := worker.QueueConsumerParams{
			NumWorkers:           conf.NumWorkers,
			QueueName:            conf.QueueName,
			Done:                 done,
			HandlerFn:            handleFn,
			KeepTasksHidden:      true,
			LogEventName:         conf.LogEventName,
			MaxVisibilityTimeout: time.Duration(conf.MaxVisibilityTimeout) * time.Minute,
			AccountID:            conf.AccountNumber,
		}

		err := validator.Validate(params)
		if err != nil {
			return nil, errx.New(err, errx.Fields{
				"worker": conf.LogEventName,
				"cause":  "failed to validate worker parameters",
			})
		}

		wg, err := worker.CreateConsumers(ctx, clients, params)
		err = errx.New(err, errx.Fields{
			"worker": conf.LogEventName,
		})
		return wg, err
	}

	// Email validaton success workers are responsible for updating the user status to verified
	// addresses that require validation
	emailValidatorWg, err := createWorker(queues.EmailVerified, emailvalidationsuccess.Run)
	if err != nil {
		return wgs, err
	}
	if emailValidatorWg != nil {
		wgs = append(wgs, emailValidatorWg)
	}

	imageUploadWg, err := createWorker(queues.ImageUpload, imageuploadqueue.Run)
	if err != nil {
		return wgs, err
	}
	if imageUploadWg != nil {
		wgs = append(wgs, imageUploadWg)
	}

	deadLetterWg, err := createWorker(queues.DeadLetter, deadletterqueue.Run)
	if err != nil {
		return wgs, err
	}
	if deadLetterWg != nil {
		wgs = append(wgs, deadLetterWg)
	}

	expireCacheWg, err := createWorker(queues.ExpireCache, expirecachequeue.Run)
	if err != nil {
		return wgs, err
	}
	if expireCacheWg != nil {
		wgs = append(wgs, expireCacheWg)
	}

	return wgs, nil
}
