package main

import (
	"net"
	"os"
	"os/signal"

	"code.justin.tv/cb/watchers/internal/api"
	"code.justin.tv/cb/watchers/internal/authorization"
	"code.justin.tv/cb/watchers/internal/backend"
	"code.justin.tv/cb/watchers/internal/datastore"
	service_common "code.justin.tv/feeds/service-common"
	_ "github.com/lib/pq"
)

const (
	teamName    = "creator-business"
	serviceName = "watchers"
)

// CodeVersion is set by the build script
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion: CodeVersion,
	},
}

type injectables struct {
	auth *authorization.Decoder
}

type service struct {
	injectables
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	configs struct {
		httpConfig api.HTTPConfig
		dbConfig   datastore.DBConfig
	}

	server    api.HTTPServer
	datastore datastore.Datastore
	backend   backend.Backender
}

func (s *service) setup() error {
	if err := s.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(
		s.serviceCommon.Config,
		&s.configs.httpConfig,
		&s.configs.dbConfig,
	); err != nil {
		return err
	}

	// Check injectables.
	if s.injectables.auth == nil {
		decoder, err := authorization.NewDecoder(s.serviceCommon.SetupAdditionalSandstormDistconf("identity", "cartman"))
		if err != nil {
			return err
		}
		s.injectables.auth = decoder
	}

	if err := s.configs.dbConfig.LoadSecrets(s.serviceCommon.Secrets); err != nil {
		return err
	}

	dbStatsSender := &service_common.StatSender{
		SubStatter:   s.serviceCommon.Statsd.NewSubStatter("db"),
		ErrorTracker: &s.serviceCommon.ErrorTracker,
	}

	datastore, err := datastore.NewDatastore(&s.configs.dbConfig, s.serviceCommon.Log, dbStatsSender)
	if err != nil {
		return err
	}
	s.datastore = datastore

	backend, err := backend.NewBackend(s.datastore, s.serviceCommon.Log)
	if err != nil {
		return err
	}
	s.backend = backend

	return nil
}

func (s *service) inject() {
	s.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &s.configs.httpConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Dims:        &s.serviceCommon.CtxDimensions,
			Log:         s.serviceCommon.Log,
			ElevateKey:  s.serviceCommon.ElevateLogKey,
			Ctxlog:      &s.serviceCommon.Ctxlog,
			OnListen:    s.onListen,
			PanicLogger: s.serviceCommon.PanicLogger,
			XRay:        s.serviceCommon.XRay,
		},
		Auth:    s.injectables.auth,
		Backend: s.backend,
		Config:  &s.configs.httpConfig,
	}

	s.server.SetupRoutes = s.server.Routes

	s.runner = service_common.ServiceRunner{
		Log: s.serviceCommon.Log,
		Services: []service_common.Service{
			&s.server,
			&s.serviceCommon,
		},
		SigChan:      s.sigChan,
		SignalNotify: signal.Notify,
	}
}

func (s *service) main() {
	if err := s.setup(); err != nil {
		if s.serviceCommon.Log != nil {
			s.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		s.osExit(1)
		return
	}
	s.inject()
	if err := s.runner.Execute(); err != nil {
		s.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.serviceCommon.Log.Log("Finished Main")
}

func main() {
	instance.main()
}
