// +build integration

package main

import (
	"context"
	"encoding/base64"
	"fmt"
	"io/ioutil"
	"math/rand"
	"net"
	"net/http"
	"os"
	"path/filepath"
	"strconv"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/cb/watchers/external/client/watchers"
	"code.justin.tv/cb/watchers/external/models"
	"code.justin.tv/cb/watchers/external/structs"
	"code.justin.tv/cb/watchers/internal/authorization"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/smartystreets/goconvey/convey"
)

const (
	serverShutdownWaitTime = 3 * time.Second
)

func timestampID() string {
	return strconv.FormatInt(time.Now().UnixNano(), 10)
}

func randomID() string {
	randID := timestampID()
	randID = randID[len(randID)-8:]
	return fmt.Sprintf("%d%s", 1+rand.Intn(9), randID)
}

func getAuthTestKey() string {
	dir, err := os.Getwd()
	if err != nil {
		panic(err)
	}
	return filepath.Join(dir, "../../internal/authorization/development_ecc_private.key")
}

func generateToken(encoder *authorization.Encoder, channelID string) *goauthorization.AuthorizationToken {
	now := time.Now()
	token := encoder.Encode(goauthorization.TokenParams{
		Exp: now.Add(10 * time.Minute),
		Nbf: now,
		Aud: []string{"code.justin.tv/cb/watchers"},
		Claims: goauthorization.CapabilityClaims{
			"view_audits": goauthorization.CapabilityClaim{
				"channel_id": channelID,
			},
		},
	})

	return token
}

func newTestDecoder() (*authorization.Decoder, error) {
	dir, err := os.Getwd()
	if err != nil {
		panic(err)
	}
	keyPath := filepath.Join(dir, "../../internal/authorization/development_ecc_public.key")
	key, err := ioutil.ReadFile(keyPath)
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("authorization: failed to read file at %s", keyPath))
	}

	decoder, err := goauthorization.NewDecoder("ES256", "code.justin.tv/cb/watchers", "code.justin.tv/web/cartman", key)
	if err != nil {
		return nil, errors.Wrap(err, "authorization: failed to instantiate decoder")
	}

	return &authorization.Decoder{decoder}, nil
}

func TestService(t *testing.T) {
	decoder, err := newTestDecoder()
	if err != nil {
		t.Error(err)
		return
	}
	ts := startServer(t, injectables{auth: decoder})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}
	channelID := randomID()
	channelIDInt, err := strconv.ParseInt(channelID, 10, 64)
	if err != nil {
		t.Error(err)
		return
	}

	var channelAudit1, channelAudit2 models.ChannelAudit
	actorID1 := "123"
	actorID2 := "345"
	objID1 := "456"
	objID2 := "456"
	action1 := structs.ActionStatusChange
	action2 := structs.ActionStatusChange

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.watchersClient
		ctx := ts.ctx

		Convey("Testing CREATE /v1/channel_audits/:channel_id", func() {
			Convey("Should return success", func() {
				oldValue := "old value"
				newValue := "new value"
				r, err := client.V1Create(ctx, structs.V1CreateRequest{
					ChannelID: channelID,
					ActorID:   actorID1,
					Action:    action1,
					ObjID:     objID1,
					OldValue:  &oldValue,
					NewValue:  &newValue,
				}, nil)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
			})
			Convey("Should return success2", func() {
				oldValue := "old value2"
				newValue := "new value2"

				r, err := client.V1Create(ctx, structs.V1CreateRequest{
					ChannelID: channelID,
					ActorID:   actorID2,
					Action:    action2,
					ObjID:     objID2,
					OldValue:  &oldValue,
					NewValue:  &newValue,
				}, nil)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
			})

		})

		Convey("Testing GET /v1/channel_audits/:channel_id", func() {
			Convey("Should return success", func() {
				createdOn := time.Now()
				before := createdOn.Add(time.Hour * 24 * 30)
				after := createdOn.Add(-1 * time.Hour * 24 * 30)

				r, err := client.V1GetAudits(ctx,
					structs.V1GetAuditsRequest{
						ChannelID: channelID,
						Limit:     10,
						Actions:   []string{structs.ActionStatusChange, "fake_action"},
						Before:    strconv.FormatInt(before.Unix(), 10),
						After:     strconv.FormatInt(after.Unix(), 10),
					}, nil)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(len(*r), ShouldEqual, 2)
				channelAudit2 = (*r)[0]
				So(channelAudit2.ChannelID, ShouldEqual, channelIDInt)
				So(channelAudit2.ObjID, ShouldEqual, 456)
				So(channelAudit2.ActorID, ShouldEqual, 345)
				So(channelAudit2.Action, ShouldEqual, action2)

				channelAudit1 = (*r)[1]
				So(channelAudit1.ChannelID, ShouldEqual, channelIDInt)
				So(channelAudit1.ObjID, ShouldEqual, 456)
				So(channelAudit1.ActorID, ShouldEqual, 123)
				So(channelAudit1.Action, ShouldEqual, action1)
			})
		})

		Convey("Testing GET /v2/channels/:channel_id/audits", func() {
			encoder, err := authorization.NewEncoder(getAuthTestKey())
			if err != nil {
				t.Errorf("Unable to setup testing server %s", err)
				return
			}

			Convey("Should return success", func() {
				token := generateToken(encoder, channelID)
				tokenString, err := token.String()
				if err != nil {
					t.Errorf("Unable to setup testing server %s", err)
					return
				}

				reqOpts := twitchclient.ReqOpts{
					ClientID:           "client_id",
					StatName:           "stat_name",
					ClientRowID:        "client_row_id",
					AuthorizationToken: tokenString,
				}
				cursor := base64.StdEncoding.EncodeToString([]byte(strconv.Itoa(channelAudit2.ID)))
				limit := 1
				newCursor := base64.StdEncoding.EncodeToString([]byte(strconv.Itoa(channelAudit1.ID)))

				r, err := client.V2GetAudits(ctx,
					structs.V2GetAuditsRequest{
						ChannelID: channelID,
						Limit:     &limit,
						Cursor:    &cursor,
					}, &reqOpts)

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(*r.Meta.Cursor, ShouldEqual, newCursor)
				So(len(r.Data), ShouldEqual, 1)
				audit := r.Data[0]
				So(audit.ChannelID, ShouldEqual, channelIDInt)
				So(audit.ObjID, ShouldEqual, channelAudit1.ObjID)
				So(audit.ActorID, ShouldEqual, channelAudit1.ActorID)
				So(audit.Action, ShouldEqual, channelAudit1.Action)
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

type testSetup struct {
	ctx            context.Context
	watchersClient watchers.Client
	httpClient     *http.Client
	host           string
	onFinish       func(timeToWait time.Duration)
	thisInstance   *service
}

func (t *testSetup) Setup() error {
	ctx, cancelFunc := context.WithTimeout(context.Background(), time.Second*5)
	t.ctx = ctx
	Reset(cancelFunc)
	client, err := watchers.NewClient(twitchclient.ClientConf{
		Host: t.host,
	})
	if err != nil {
		return err
	}
	t.watchersClient = client
	t.httpClient = &http.Client{}
	return nil
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func startServer(t *testing.T, i injectables, configs ...map[string][]byte) *testSetup {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"watchers.listen_addr": []byte(":0"),
		"rollbar.access_token": []byte(""),
		"statsd.hostport":      []byte(""),
		"debug.addr":           []byte(":0"),
		"logging.to_stdout":    []byte("false"),
		"logging.to_stderr":    []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return nil
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			t.Error(err)
			return nil
		}
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger: panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 35):
		t.Error("Took to long to start service")
		return nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testSetup{
		host:         addressForIntegrationTests,
		onFinish:     onFinish,
		thisInstance: &thisInstance,
	}
}
