// +build integration

package main

import (
	"context"
	"fmt"
	"math/rand"
	"net"
	"net/http"
	"os"
	"strconv"
	"strings"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/cb/watchers/external/models"
	"code.justin.tv/cb/watchers/external/structs"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/smartystreets/goconvey/convey"
)

const (
	serverShutdownWaitTime = 3 * time.Second
)

func timestampID() string {
	return strconv.FormatInt(time.Now().UnixNano(), 10)
}

func randomID() string {
	randID := timestampID()
	randID = randID[len(randID)-8:]
	return fmt.Sprintf("%d%s", 1+rand.Intn(9), randID)
}

func TestService(t *testing.T) {
	ts := startServer(t)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}
	channelID := randomID()
	channelIDInt, err := strconv.ParseInt(channelID, 10, 64)
	if err != nil {
		panic(err)
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		ctx := ts.ctx

		Convey("Testing POST /process", func() {
			oldValue := "sdfasdf"
			newValue := "sdfdsfsd"
			ts.thisInstance.datastore.CreateAudit(ctx, models.ChannelAudit{
				ChannelID: channelIDInt,
				Action:    structs.ActionStatusChange,
				OldValue:  &oldValue,
				NewValue:  &newValue,
				ActorID:   123,
				ObjID:     123,
			})

			message := fmt.Sprintf(`{"message":"{\"user_id\":\"%s\",\"timestamp\":\"%s\"}"}`,
				channelID, time.Now().Format(time.RFC3339))
			body := strings.NewReader(message)

			req, err := ts.client.NewRequest("POST", "/process", body)
			So(err, ShouldBeNil)

			_, err = ts.client.DoNoContent(ctx, req, twitchclient.ReqOpts{})

			// Response should be nil
			So(err, ShouldBeNil)

			// No Audits should be present
			list, err := ts.thisInstance.datastore.GetAudits(ctx, channelID, 10, nil, []string{}, nil, nil)

			So(err, ShouldBeNil)
			So(len(list), ShouldEqual, 0)
		})

	})

	ts.onFinish(serverShutdownWaitTime)
}

type testSetup struct {
	ctx          context.Context
	httpClient   *http.Client
	client       twitchclient.Client
	host         string
	onFinish     func(timeToWait time.Duration)
	thisInstance *service
}

func (t *testSetup) Setup() error {
	ctx, cancelFunc := context.WithTimeout(context.Background(), time.Second*5)
	t.ctx = ctx
	Reset(cancelFunc)
	client, err := twitchclient.NewClient(twitchclient.ClientConf{
		Host: t.host,
	})
	if err != nil {
		return err
	}
	t.httpClient = &http.Client{}
	t.client = client
	return nil
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func startServer(t *testing.T, configs ...map[string][]byte) *testSetup {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"watchers.listen_addr": []byte(":0"),
		"rollbar.access_token": []byte(""),
		"statsd.hostport":      []byte(""),
		"debug.addr":           []byte(":0"),
		"logging.to_stdout":    []byte("false"),
		"logging.to_stderr":    []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return nil
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			t.Error(err)
			return nil
		}
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger: panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 35):
		t.Error("Took to long to start service")
		return nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testSetup{
		host:         addressForIntegrationTests,
		onFinish:     onFinish,
		thisInstance: &thisInstance,
	}
}
