package watchers

import (
	"context"
	"fmt"
	"net/url"
	"strconv"

	"code.justin.tv/cb/watchers/external/structs"
	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/twitchdoer"
	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultTimingXactName = "cb-watchers"
	defaultStatPrefix     = "service.cb_watchers."
	defaultStatSampleRate = 1.0
)

// Client satisfies visage's interface requirements for clients
type Client interface {
	V1Create(ctx context.Context, input structs.V1CreateRequest, reqOpts *twitchclient.ReqOpts) (*structs.V1CreateResponse, error)
	V1GetAudits(ctx context.Context, input structs.V1GetAuditsRequest, reqOpts *twitchclient.ReqOpts) (*structs.V1GetAuditsResponse, error)
	V2GetAudits(ctx context.Context, input structs.V2GetAuditsRequest, reqOpts *twitchclient.ReqOpts) (*structs.V2GetAuditsResponse, error)
}

// ClientImpl implements the Client interface and uses the twitchhttp client to make http requests
type ClientImpl struct {
	Client twitchclient.Client
}

// NewClient returns a new instance of the Client which uses the given config
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &ClientImpl{Client: twitchClient}, nil
}

func (c *ClientImpl) http(ctx context.Context, statName string, method string, path string, queryParams url.Values, body interface{}, reqOpts *twitchclient.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       defaultStatPrefix + statName,
		StatSampleRate: defaultStatSampleRate,
	})

	doer := &twitchdoer.TwitchHTTPDoer{
		Client: c.Client,
		Reqopt: combinedReqOpts,
	}
	return clients.DoHTTP(ctx, doer, method, path, queryParams, body, into, doer.NewTwitchRequest)
}

// V1Create ...
func (c *ClientImpl) V1Create(ctx context.Context, input structs.V1CreateRequest, reqOpts *twitchclient.ReqOpts) (*structs.V1CreateResponse, error) {
	path := fmt.Sprintf("/v1/channel_audits/%s", input.ChannelID)

	var response structs.V1CreateResponse
	err := c.http(ctx, "v1_create", "POST", path, nil, input, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

// V1GetAudits ...
func (c *ClientImpl) V1GetAudits(ctx context.Context, input structs.V1GetAuditsRequest, reqOpts *twitchclient.ReqOpts) (*structs.V1GetAuditsResponse, error) {
	path := fmt.Sprintf("/v1/channel_audits/%s", input.ChannelID)
	query := url.Values{}
	query.Add("limit", strconv.Itoa(input.Limit))
	for _, value := range input.Actions {
		query.Add("actions[]", value)
	}
	query.Add("cursor", input.Cursor)
	query.Add("before", input.Before)
	query.Add("after", input.After)

	var response structs.V1GetAuditsResponse
	err := c.http(ctx, "v1_get_channel_audits", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

// V2GetAudits ...
func (c *ClientImpl) V2GetAudits(ctx context.Context, input structs.V2GetAuditsRequest, reqOpts *twitchclient.ReqOpts) (*structs.V2GetAuditsResponse, error) {
	path := fmt.Sprintf("/v2/channels/%s/audits", input.ChannelID)

	query := url.Values{}
	if input.Limit != nil {
		query.Add("limit", strconv.Itoa(*input.Limit))
	}

	if input.Cursor != nil {
		query.Add("cursor", *input.Cursor)
	}

	var response structs.V2GetAuditsResponse
	err := c.http(ctx, "v2_get_channel_audits", "GET", path, query, nil, reqOpts, &response)
	if err != nil {
		return nil, err
	}
	return &response, nil
}
