package api

import (
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/cb/watchers/external/structs"
	"code.justin.tv/feeds/service-common"
	"goji.io/pat"
)

func (s *HTTPServer) v1Get(r *http.Request) (interface{}, error) {
	params, err := v1GetParams(r)
	if err != nil {
		return nil, err
	}

	audits, err := s.Backend.V1GetAudits(r.Context(), *params)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  fmt.Errorf("error on fetch audits: %s", err),
		}
	}

	return &cachedResponse{
		payload:   structs.V1GetAuditsResponse(audits),
		cacheTime: 1 * time.Minute,
	}, nil
}

func v1GetParams(r *http.Request) (*structs.V1GetAuditsRequest, error) {
	channelID := pat.Param(r, "channel_id")
	if channelID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("channel id is required"),
		}
	}

	limit := r.URL.Query().Get("limit")

	if limit == "" {
		limit = "10"
	}

	limitInt, err := strconv.Atoi(limit)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("limit should be integer"),
		}
	}

	actions := r.URL.Query()["actions[]"]

	cursor := r.URL.Query().Get("cursor")

	before := transformEpochToSQLTime(r.URL.Query().Get("before"))
	after := transformEpochToSQLTime(r.URL.Query().Get("after"))

	return &structs.V1GetAuditsRequest{
		ChannelID: channelID,
		Limit:     limitInt,
		Cursor:    cursor,
		Actions:   actions,
		Before:    before,
		After:     after,
	}, nil
}

func transformEpochToSQLTime(before string) string {
	now := time.Now().Format(structs.SQLTimeFormat)
	if before == "" {
		return now
	}

	beforeInt, err := strconv.ParseInt(before, 10, 64)
	if err != nil {
		return now
	}

	return time.Unix(beforeInt, 0).Format(structs.SQLTimeFormat)
}
