package api

import (
	"encoding/base64"
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/cb/watchers/external/structs"
	service_common "code.justin.tv/feeds/service-common"
	"goji.io/pat"
)

func (s *HTTPServer) v2Get(r *http.Request) (interface{}, error) {
	var cursor string

	params, err := v2GetParams(r)
	if err != nil {
		return nil, err
	}

	if err = isAuthorizedChannelID(r, s.Auth, params.ChannelID); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusForbidden,
			Err:  fmt.Errorf("authorization fail %s", err),
		}
	}
	audits, err := s.Backend.V2GetAudits(r.Context(), *params)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  fmt.Errorf("error on fetch audits: %s", err),
		}
	}
	if len(audits) > 0 {
		cursor = base64.StdEncoding.EncodeToString([]byte(strconv.Itoa(audits[len(audits)-1].ID)))
	}

	response := structs.V2GetAuditsResponse{
		Meta: structs.V2GetAuditsMeta{
			Limit:  *params.Limit,
			Cursor: &cursor,
		},
		Data: audits,
	}

	return &cachedResponse{
		payload:   response,
		cacheTime: 1 * time.Minute,
	}, nil
}

func v2GetParams(r *http.Request) (*structs.V2GetAuditsRequest, error) {
	minGetLimit := 1
	maxGetLimit := 50

	var (
		cursor string
		err    error
	)

	channelID := pat.Param(r, "channel_id")
	if channelID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("channel id is required"),
		}
	}

	limit := maxGetLimit
	limitParam := r.URL.Query().Get("limit")

	if limitParam != "" {
		limit, err = strconv.Atoi(limitParam)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  fmt.Errorf("limit should be integer %s", err),
			}
		}
		if limit < minGetLimit || limit > maxGetLimit {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  fmt.Errorf("limit should be between %d-%d", minGetLimit, maxGetLimit),
			}
		}
	}

	cursorParam := r.URL.Query().Get("cursor")
	if cursorParam != "" {
		cursor = cursorParam
	}

	return &structs.V2GetAuditsRequest{
		ChannelID: channelID,
		Limit:     &limit,
		Cursor:    &cursor,
	}, nil
}
