package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/cb/watchers/external/models"
	"code.justin.tv/feeds/errors"
)

func (d *datastore) CreateAudit(ctx context.Context, input models.ChannelAudit) (*models.ChannelAudit, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "createAudit", startTime, err == nil)
	}(time.Now())

	query := `
		INSERT INTO channel_audits (
			channel_id, actor_id, obj_id,
			action, new_value, old_value, created_on, created_at_nanos, audit_type
		) VALUES (
			$1, $2, $3, $4, $5, $6, $7, $8, $9
		) RETURNING
		  id, channel_id, actor_id, obj_id, action, new_value, old_value, created_on, created_at_nanos, audit_type`

	row := d.QueryRowContext(
		ctx,
		query,
		input.ChannelID,
		input.ActorID,
		input.ObjID,
		input.Action,
		input.NewValue,
		input.OldValue,
		now(),
		time.Now().UTC().UnixNano(),
		input.AuditType,
	)

	audit := models.ChannelAudit{}
	err = row.Scan(
		&audit.ID,
		&audit.ChannelID,
		&audit.ActorID,
		&audit.ObjID,
		&audit.Action,
		&audit.NewValue,
		&audit.OldValue,
		&audit.CreatedOn,
		&audit.CreatedAtNanos,
		&audit.AuditType,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not create %v", input)
	}

	return &audit, nil
}
