package datastore

import (
	"context"
	"database/sql"
	"fmt"

	"code.justin.tv/cb/watchers/external/models"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
)

// Datastore represents the interface for performing data operations.
type Datastore interface {
	CreateAudit(ctx context.Context, model models.ChannelAudit) (*models.ChannelAudit, error)
	GetAudits(ctx context.Context, channelID string, limit int, cursor *string, actions []string,
		before *string, after *string) ([]models.ChannelAudit, error)
	DeleteAuditsByUserID(ctx context.Context, channelID string) error
}

// datastore contains the sql DB connection.
type datastore struct {
	*sql.DB

	log   *log.ElevatedLog
	stats *service_common.StatSender
}

// NewDatastore initializes a datastore to use with sql DB connection.
func NewDatastore(config *DBConfig, logger *log.ElevatedLog, stats *service_common.StatSender) (Datastore, error) {
	logger.Log("hostname", config.hostname.Get(), "Opening connection to Postgres")

	connectionString := fmt.Sprintf("postgres://%s:%s@%s", config.username.Get(), config.password.Get(), config.hostname.Get())
	db, err := sql.Open(driver, connectionString)
	if err != nil {
		return nil, err
	}

	db.SetMaxOpenConns(int(config.maxOpenConns.Get()))

	return &datastore{
		DB:    db,
		log:   logger,
		stats: stats,
	}, nil
}
