package datastore

import (
	"context"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
)

const driver = "postgres"

// DBConfig ...
type DBConfig struct {
	hostname     *distconf.Str
	username     *distconf.Str
	password     *distconf.Str
	maxOpenConns *distconf.Int
}

// Load ...
func (c *DBConfig) Load(d *distconf.Distconf) error {
	c.hostname = d.Str("db.hostname", "")
	if c.hostname.Get() == "" {
		return errors.New("db hostname could not be loaded")
	}

	c.maxOpenConns = d.Int("db.max_open_connections", 0)
	if c.maxOpenConns.Get() == 0 {
		return errors.New("db max_open_connections could not be loaded")
	}
	return nil
}

// LoadSecrets ...
func (c *DBConfig) LoadSecrets(d *distconf.Distconf) error {
	c.username = d.Str("db.username", "")
	c.password = d.Str("db.password", "")

	if c.username.Get() == "" || c.password.Get() == "" {
		return errors.New("db username and password could not be loaded")
	}
	return nil
}

func (d *datastore) recordStats(ctx context.Context, operationName string, startTime time.Time, succeeded bool) {
	// Record duration so that we can calculate latency.
	endTime := time.Now()
	duration := endTime.Sub(startTime)
	d.stats.TimingDurationC(operationName+".time", duration, 1)

	// Distinguish between the operation succeeding, failing due the context being canceled, and failing due to a db
	// error.
	status := "success"
	if !succeeded {
		if ctx.Err() != nil {
			status = "ctx_error"
		} else {
			status = "db_error"
		}
	}

	// Record count so that we can calculate throughput.
	d.stats.IncC(operationName+".status."+status, 1, 1)
}

func now() time.Time {
	return ConvertToDBTime(time.Now())
}

// ConvertToDBTime converts the given time to a time that can be reliably represented by Postgres.
func ConvertToDBTime(t time.Time) time.Time {
	// Ensure that we're using a UTC time to avoid issues with serializing/deserializing timezone info to
	// the database.
	t = t.UTC()

	// Unlike Go, Postgres date/time types only support a resolution of up to 1 microsecond.  Trim the given time
	// so we can work with a time that the database can represent.
	return t.Truncate(time.Microsecond)
}
