package datastore

import (
	"context"
	"database/sql"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/cb/watchers/external/models"
	"code.justin.tv/cb/watchers/external/structs"
	"code.justin.tv/feeds/errors"
)

func (d *datastore) GetAudits(ctx context.Context,
	channelID string, limit int, cursor *string, actions []string,
	before *string, after *string) ([]models.ChannelAudit, error) {
	allowedActions := [6]string{structs.ActionCommercial, structs.ActionCut,
		structs.ActionAddEditor, structs.ActionRemoveEditor,
		structs.ActionGameChange, structs.ActionStatusChange}

	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "getAudits", startTime, err == nil)
	}(time.Now())

	var values []interface{}
	var where []string
	whereCounter := 0

	//Adding where statement components
	whereCounter++
	values = append(values, channelID)
	where = append(where, fmt.Sprintf("channel_id = $%d", whereCounter))

	if after != nil {
		whereCounter++
		values = append(values, after)
		where = append(where, fmt.Sprintf("created_on >= $%d", whereCounter))
	}
	if before != nil {
		whereCounter++
		values = append(values, before)
		where = append(where, fmt.Sprintf("created_on <= $%d", whereCounter))
	}

	// Adding cursor where statement
	if cursor != nil && *cursor != "" {
		whereCounter++
		values = append(values, *cursor)
		where = append(where, fmt.Sprintf("id < $%d", whereCounter))
	}

	// Filter Actions and build where statement
	if len(actions) > 0 {
		filteredActions := []string{}
		for _, action := range actions {
			for _, allowedAction := range allowedActions {
				if allowedAction == action {
					filteredActions = append(filteredActions, fmt.Sprintf("'%s'", action))
				}
			}
		}
		if len(filteredActions) > 0 {
			actionsString := strings.Join(filteredActions, ",")
			where = append(where, fmt.Sprintf("action IN(%s)", actionsString))
		}
	}

	// Apply actions and cursor `where` conditions.
	query := `
		SELECT id, channel_id, actor_id, obj_id, action, new_value, old_value, created_on
		FROM channel_audits
		WHERE
			%s
		ORDER BY id DESC
		LIMIT %d
	`
	query = fmt.Sprintf(query, strings.Join(where, " AND "), limit)
	rows, err := d.QueryContext(
		ctx,
		query,
		values...)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not fetch audits %s(%s)", query, values)
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			d.log.Log(err)
		}
	}()

	result := []models.ChannelAudit{}
	for rows.Next() {
		audit := models.ChannelAudit{}
		err = rows.Scan(
			&audit.ID,
			&audit.ChannelID,
			&audit.ActorID,
			&audit.ObjID,
			&audit.Action,
			&audit.NewValue,
			&audit.OldValue,
			&audit.CreatedOn,
		)
		if err != nil {
			return nil, errors.Wrap(err, "db: failed to scan editable rows")
		}

		result = append(result, audit)
	}

	err = rows.Err()
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate editable rows")
	}

	return result, nil
}
