package statsdtruth

import (
	"sync"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
)

// StatsdTruth sends pre-determined metrics as a way to verify the statsd receiving metrics system is working correctly.
// By default, every 200ms it will increment a counter and send a duration.  The duration goes steadily from 0ms to 1s,
// in 100ms increments, before wrapping back over.
type StatsdTruth struct {
	SendTo   statsd.StatSender
	Interval time.Duration

	once        sync.Once
	closeSignal chan struct{}
}

func (s *StatsdTruth) interval() time.Duration {
	if s.Interval == 0 {
		return time.Millisecond * 200
	}
	return s.Interval
}

func (s *StatsdTruth) setup() {
	s.closeSignal = make(chan struct{})
}

// Start sending metrics in a loop
func (s *StatsdTruth) Start() error {
	s.once.Do(s.setup)
	nextDuration := time.Duration(0)
	ticker := time.NewTicker(s.interval())
	loopCount := int64(0)
	defer ticker.Stop()
	for {
		select {
		case <-s.closeSignal:
			return nil
		case <-ticker.C:
			s.SendTo.Inc("count", 1, 1.0)
			s.SendTo.TimingDuration("timing", nextDuration, 1.0)
			s.SendTo.Gauge("gauge", loopCount, 1.0)
			loopCount++
			nextDuration += time.Millisecond * 100
			if nextDuration >= time.Second {
				nextDuration = 0
			}
		}
	}
}

// Close ends metric sending
func (s *StatsdTruth) Close() error {
	s.once.Do(s.setup)
	close(s.closeSignal)
	return nil
}
