package service_common

import (
	"net"
	"net/http"
	"time"

	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/xray"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"
)

// Verify config from distconf
func (c *BaseHTTPServerConfig) Verify(dconf *distconf.Distconf, prefix string) error {
	c.ListenAddr = dconf.Str(prefix+".listen_addr", ":8000")
	c.ShutdownWait = dconf.Duration(prefix+".shutdown_wait", time.Second*5)
	return nil
}

// BaseHTTPServerConfig configures common HTTP parameters
type BaseHTTPServerConfig struct {
	// ListenAddr is the address to listen on
	ListenAddr *distconf.Str
	// ShutdownWait is the timeout for graceful shutdown
	ShutdownWait *distconf.Duration
}

// BaseHTTPServer is the base embedded struct all feed HTTP structs should include
type BaseHTTPServer struct {
	// Config controls configuration for the server
	Config *BaseHTTPServerConfig
	// SetupRoutes is required and where the goji mux's routes should be configured
	SetupRoutes func(mux *goji.Mux)
	// Stats control graphite stats
	Stats *StatSender
	// Log allows logging
	Log *log.ElevatedLog
	// EleveateKey is the used by ElevateLog to elevate debug logs
	ElevateKey interface{}
	// Ctxlog allows logging with ctx
	Ctxlog *ctxlog.Ctxlog
	// PanicLogger logs HTTP panics
	PanicLogger PanicLogger
	// Dims allows optional dimensions on contexts
	Dims *log.CtxDimensions
	// XRay allows tracing requests
	XRay *xray.XRay `nilcheck:"nodepth"`

	// OnListen is an optional callback for when a valid address is open
	OnListen func(listenAddr net.Addr) `nilcheck:"ignore"`
	// HealthCheck is an optional function to check if the server is healthy
	HealthCheck func(req *http.Request) error `nilcheck:"ignore"`

	listener net.Listener
	server   http.Server
}

// Setup creates the basic chitin context for the server
func (s *BaseHTTPServer) Setup() error {
	var err error
	s.listener, err = net.Listen("tcp", s.Config.ListenAddr.Get())
	if err != nil {
		return errors.Wrap(err, "unable to setup listening port")
	}
	mux := CreateDefaultMux(s.Log, s.ElevateKey, s.Ctxlog, s.PanicLogger, s.Dims, s.XRay)
	mux.Handle(pat.Get("/debug/health"), s.CreateHandler("health_check", s.healthCheck))
	s.SetupRoutes(mux)
	s.server = http.Server{
		Handler: mux,
	}
	if s.OnListen != nil && err == nil {
		s.OnListen(s.listener.Addr())
	}
	if _, err := s.healthCheck(&http.Request{}); err != nil {
		return err
	}
	return nil
}

// CreateHandler returns a HTTP handler for goji that works like our standard handlers
func (s *BaseHTTPServer) CreateHandler(name string, callback func(req *http.Request) (interface{}, error)) *JSONHandler {
	return &JSONHandler{
		Log:          s.Log,
		Stats:        s.Stats.NewSubStatSender(name),
		ItemCallback: callback,
	}
}

func (s *BaseHTTPServer) healthCheck(r *http.Request) (interface{}, error) {
	if s.HealthCheck != nil {
		if err := s.HealthCheck(r); err != nil {
			return nil, err
		}
	}
	return "OK", nil
}

// Start serves requests on the listening port
func (s *BaseHTTPServer) Start() error {
	s.Log.Log("addr", s.listener.Addr().String(), "starting listen on server")
	err := s.server.Serve(s.listener)
	s.Log.Log("err", err, "server finished listening")
	return err
}

// Close stops the running http server
func (s *BaseHTTPServer) Close() error {
	shutdownCtx, cancel := context.WithTimeout(context.Background(), s.Config.ShutdownWait.Get())
	defer cancel()
	return s.server.Shutdown(shutdownCtx)
}
