package service_common

import (
	"fmt"
	"reflect"

	"io"
	"strings"
	"sync"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"golang.org/x/net/context"
)

// StackLogger checks for any err keys and wraps them if needed so that we get a pretty stack trace in the finished log
// because the stack trace is eaten when we pass the log onto a channel
type StackLogger struct {
	RootLogger log.Logger
}

func (s *StackLogger) Log(keyvals ...interface{}) {
	for idx := 0; idx+1 < len(keyvals); idx += 2 {
		if mapKey(keyvals[idx]) == "err" {
			if err, ok := keyvals[idx+1].(error); ok {
				keyvals[idx+1] = wrapErrIfNeeded(err)
			}
		}
	}
	s.RootLogger.Log(keyvals...)
}

func mapKey(k interface{}) (s string) {
	defer func() {
		if panicVal := recover(); panicVal != nil {
			s = nilCheck(k, panicVal, "NULL").(string)
		}
	}()
	switch x := k.(type) {
	case string:
		return x
	case fmt.Stringer:
		return x.String()
	default:
		return fmt.Sprint(x)
	}
}

func nilCheck(ptr, panicVal interface{}, onError interface{}) interface{} {
	if vl := reflect.ValueOf(ptr); vl.Kind() == reflect.Ptr && vl.IsNil() {
		return onError
	}
	panic(panicVal)
}

func wrapErrIfNeeded(err error) error {
	type tracedError1 interface {
		StackTrace() []uintptr
	}
	type tracedError2 interface {
		Stack() []uintptr
	}
	if _, ok := err.(tracedError1); ok {
		return err
	}
	if _, ok := err.(tracedError2); ok {
		return err
	}
	return errors.Wrap(err, "")
}

type substringWatch struct {
	toFind       string
	hasFoundLine chan struct{}
	parent       *LineOutputWatcher
}

var _ lineChecker = &substringWatch{}

func (s *substringWatch) CheckLine(line string) bool {
	if strings.Contains(line, s.toFind) {
		close(s.hasFoundLine)
		return true
	}
	return false
}

func (s *substringWatch) Block(ctx context.Context) bool {
	defer s.parent.removeChecker(s)

	// Always prefer having found the line if it was found
	select {
	case <-s.hasFoundLine:
		return true
	default:
	}

	select {
	case <-ctx.Done():
		return false
	case <-s.hasFoundLine:
		return true
	}
}

type lineChecker interface {
	CheckLine(line string) bool
}

// ConditionWaiter is something that can block for a condition to happen
type ConditionWaiter interface {
	Block(ctx context.Context) bool
}

// LineOutputWatcher can forward output to another writer, while allowing people to register watches for lines in the
// output.  This is good for ensuring that a logger eventually splits out a message
type LineOutputWatcher struct {
	Out io.Writer

	mu           sync.Mutex
	lineBuffer   string
	lineCheckers []lineChecker
}

var _ io.Writer = &LineOutputWatcher{}

// Write forwards the request to Out and checks inner lines
func (l *LineOutputWatcher) Write(p []byte) (n int, err error) {
	l.mu.Lock()
	defer l.mu.Unlock()
	l.lineBuffer += string(p)
	for {
		lineIndex := strings.IndexByte(l.lineBuffer, '\n')
		if lineIndex == -1 {
			break
		}
		var line string
		line, l.lineBuffer = l.lineBuffer[0:lineIndex+1], l.lineBuffer[lineIndex+1:]
		newLineCheckers := l.lineCheckers[:0]
		for _, checker := range l.lineCheckers {
			shouldRemove := checker.CheckLine(line)
			if !shouldRemove {
				newLineCheckers = append(newLineCheckers, checker)
			}
		}
		l.lineCheckers = newLineCheckers
	}
	return l.Out.Write(p)
}

func (l *LineOutputWatcher) removeChecker(toRemove lineChecker) {
	l.mu.Lock()
	defer l.mu.Unlock()
	for i, c := range l.lineCheckers {
		if c == toRemove {
			l.lineCheckers = append(l.lineCheckers[:i], l.lineCheckers[i:]...)
			return
		}
	}
}

// FindFirstSubstr returns a condition that can be blocked waiting for a substring to enter the log output
func (l *LineOutputWatcher) FindFirstSubstr(substr string) ConditionWaiter {
	l.mu.Lock()
	defer l.mu.Unlock()
	s := &substringWatch{
		toFind:       substr,
		hasFoundLine: make(chan struct{}),
		parent:       l,
	}
	l.lineCheckers = append(l.lineCheckers, s)
	return s
}
