import * as React from 'react';
import { find } from 'lodash-es';

import { queryEvolutionSets, queryUser } from 'common/helpers';
import { Evolution, EvolutionSet, Badge, Page, PAGES } from 'common/constants';
import { LoadingPage } from 'ui/pages/Loading';
import { SelectPage } from 'ui/pages/Select';
import { ProgressPage } from 'ui/pages/Progress';

require('common/styles/app.scss');

export interface AppState {
  user_id: string;
  availableEvolutions: any;
  status: any;
  loading: boolean;
  page: Page;
}

export class App extends React.Component<{}, AppState> {
  public constructor(props: any) {
    super(props);
    this.state = {
      user_id: '1',
      availableEvolutions: undefined,
      status: undefined,
      loading: true,
      page: PAGES.LOADING,
    };
  }

  public componentDidMount() {
    this.queryApi();
  }

  public render() {
    const { user_id, availableEvolutions, loading, page, status } = this.state;
    let content: JSX.Element;
    if (page === PAGES.LOADING) {
      content = <LoadingPage />;
    } else if (page === PAGES.SELECT) {
      const props = {
        badges: status.badges,
        setBadge: this.setBadge.bind(this)
      };
      content = <SelectPage {...props} />;
    } else {
      const props = {
        ...this.getCurrentInfo(),
        toSelect: this.toPage.bind(this, PAGES.SELECT)
      };
      content = <ProgressPage {...props} />;
    }

    return (
      <div className="test-container">
        <div className="extension-container">
          {content}
        </div>
      </div>
    );
  }

  private queryApi() {
    Promise
      .all([
        queryUser(this.state.user_id),
        queryEvolutionSets(),
      ])
      .then(([userData, evolutionSets]) => {
        const selecting = userData.badges.length === 0;
        this.setState({
          status: userData,
          availableEvolutions: evolutionSets,
          loading: false,
          page: selecting ? PAGES.SELECT : PAGES.PROGRESS,
        });
      });
  }

  private getCurrentInfo() {
    const { status } = this.state;
    const currentBadge = find(status.badges, (badge: Badge) => {
      return badge.selected;
    });
    const currentEvolution = find(status.evolutions, (evo: Evolution) => {
      return evo.selected;
    });

    return { currentBadge, currentEvolution };
  }

  private toPage(page: Page) {
    this.setState({
      page
    });
  }

  private setBadge(selectedBadge: Badge) {
    const { status } = this.state;
    let statusUpdate = { ...status };
    statusUpdate.badges.forEach((badge: Badge) => {
      if (badge.selected) {
        delete badge.selected;
      }
      if (badge.id === selectedBadge.id) {
        badge.selected = true;
      }
    });
    statusUpdate.evolutions.forEach((evo: Evolution) => {
      if (evo.selected) {
        delete evo.selected;
      }
      if (evo.id === selectedBadge.evolution) {
        evo.selected = true;
      }
    });
    this.setState({
      status: statusUpdate,
      page: PAGES.PROGRESS
    });
  }
}
