package main

import (
	"fmt"
	"os"
	"strconv"

	"github.com/gin-gonic/gin"
	log "github.com/sirupsen/logrus"

	"code.justin.tv/content/evo/app"
	"code.justin.tv/content/evo/model"
	"code.justin.tv/content/evo/util"
)

func main() {
	app.InitTextLog()
	log.SetLevel(log.InfoLevel)

	db, err := app.InitDB()
	if err != nil {
		log.Fatal(err)
	}

	port, err := util.Atoi(os.Getenv("PORT"), 3000)
	if err != nil {
		log.Fatalf("invalid port: %q", os.Getenv("PORT"))
	}

	startAPI := InitAPI(db)
	log.Fatal(startAPI(port))
}

func ginErr(ctx *gin.Context, fmtStr string, args ...interface{}) {
	ctx.JSON(500, gin.H{"error": fmt.Sprintf(fmtStr, args...)})
}

func InitAPI(db *model.EvoDB) func(int) error {
	// TODO: split off handlers so that they're not in one giant function

	engine := gin.Default()

	engine.GET("/health", func(ctx *gin.Context) { ctx.Status(200) })

	// Handlers for static metadata about evolutions, sets, and badges.

	engine.GET("/evolutions", func(ctx *gin.Context) {
		if evolutions, err := db.ListEvolutions(); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, evolutions)
		}
	})

	engine.GET("/evolutions/:evolution_id", func(ctx *gin.Context) {
		idStr := ctx.Params.ByName("evolution_id")
		if id, err := strconv.Atoi(idStr); err != nil {
			ginErr(ctx, "invalid evolution id: %q", idStr)
		} else if evolution, err := db.FindEvolution(id); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, evolution)
		}
	})

	engine.GET("/evolutions/:evolution_id/sets", func(ctx *gin.Context) {
		idStr := ctx.Params.ByName("evolution_id")
		if id, err := strconv.Atoi(idStr); err != nil {
			ginErr(ctx, "invalid evolution id: %q", idStr)
		} else if evolutionSets, err := db.ListEvolutionSets(id); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, evolutionSets)
		}
	})

	engine.GET("/evolutions/:evolution_id/sets/:evolution_set_id", func(ctx *gin.Context) {
		// Don't actually need evolution_id for this, but make API consistent.
		evoIDStr := ctx.Params.ByName("evolution_id")
		evoSetIDStr := ctx.Params.ByName("evolution_set_id")
		if evoID, err := strconv.Atoi(evoIDStr); err != nil {
			ginErr(ctx, "invalid evolution id: %q", evoIDStr)
		} else if evoSetID, err := strconv.Atoi(evoSetIDStr); err != nil {
			ginErr(ctx, "invalid evolution set id: %q", evoSetIDStr)
		} else if evolutionSet, err := db.FindEvolutionSet(evoID); err != nil {
			ginErr(ctx, err.Error())
		} else if evolutionSet.EvolutionID != evoID {
			ginErr(ctx, "evolution %d has no evolution set with id %d", evoID, evoSetID)
		} else {
			ctx.JSON(200, evolutionSet)
		}
	})

	engine.POST("/evolutions/sync", func(ctx *gin.Context) {
		if err := db.SyncEvoCache(); err != nil {
			ginErr(ctx, err.Error())
		} else if evolutions, err := db.ListEvolutions(); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, gin.H{"evolutions": len(evolutions)})
		}
	})

	// Handlers for user-specific requests.

	engine.GET("/users/:id", func(ctx *gin.Context) {
		opaqueID := ctx.Params.ByName("id")
		if user, err := db.FindOrCreateUser(opaqueID); err != nil {
			log.Errorf("error getting user %q: %s", opaqueID, err)
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, user)
		}
	})

	engine.GET("/users/:id/sets", func(ctx *gin.Context) {
		opaqueID := ctx.Params.ByName("id")
		if user, err := db.FindOrCreateUser(opaqueID); err != nil {
			log.Errorf("error getting user %q: %s", opaqueID, err)
			ginErr(ctx, err.Error())
		} else if evoIDStr, found := ctx.GetQuery("evolution_id"); !found {
			ginErr(ctx, "missing query parameter: evolution_id")
		} else if evoID, err := strconv.Atoi(evoIDStr); err != nil {
			ginErr(ctx, "invalid evolution_id: %q", evoIDStr)
		} else if evoSets, err := user.AvailableEvolutionSets(evoID); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, evoSets)
		}
	})

	engine.GET("/users/:id/sets/active", func(ctx *gin.Context) {
		opaqueID := ctx.Params.ByName("id")
		if user, err := db.FindOrCreateUser(opaqueID); err != nil {
			log.Errorf("error getting user %q: %s", opaqueID, err)
			ginErr(ctx, err.Error())
		} else if evoSet, err := user.ActiveEvolutionSet(); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, evoSet)
		}
	})

	engine.GET("/users/:id/sets/:set_id", func(ctx *gin.Context) {
		opaqueID := ctx.Params.ByName("id")
		evoSetIDStr := ctx.Params.ByName("set_id")
		if user, err := db.FindOrCreateUser(opaqueID); err != nil {
			log.Errorf("error getting user %q: %s", opaqueID, err)
			ginErr(ctx, err.Error())
		} else if evoSetID, err := strconv.Atoi(evoSetIDStr); err != nil {
			ginErr(ctx, err.Error())
		} else if evoSet, err := user.FindEvolutionSet(evoSetID); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, evoSet)
		}
	})

	engine.POST("/users/:id/sets/active", func(ctx *gin.Context) {
		opaqueID := ctx.Params.ByName("id")
		if user, err := db.FindOrCreateUser(opaqueID); err != nil {
			log.Errorf("error getting user %q: %s", opaqueID, err)
			ginErr(ctx, err.Error())
		} else if evoSetIDStr, found := ctx.GetQuery("evolution_set_id"); !found {
			ginErr(ctx, "missing query parameter: evolution_set_id")
		} else if evoSetID, err := strconv.Atoi(evoSetIDStr); err != nil {
			ginErr(ctx, err.Error())
		} else if evoSet, err := user.UpdateEvolutionSet(evoSetID); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.JSON(200, evoSet)
		}
	})

	// placeholder endpoint for testing
	engine.POST("/users/:id/xp", func(ctx *gin.Context) {
		opaqueID := ctx.Params.ByName("id")
		if user, err := db.FindOrCreateUser(opaqueID); err != nil {
			log.Errorf("error getting user %q: %s", opaqueID, err)
			ginErr(ctx, err.Error())
		} else if err = user.AddXP(); err != nil {
			ginErr(ctx, err.Error())
		} else {
			ctx.Status(200)
		}
	})

	return func(port int) error { return engine.Run("0.0.0.0:" + strconv.Itoa(port)) }
}
