package main

import (
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	log "github.com/sirupsen/logrus"

	"code.justin.tv/content/evo/app"
	"code.justin.tv/content/evo/events"
	"code.justin.tv/content/evo/model"
)

func main() {
	app.InitTextLog()
	log.SetLevel(log.InfoLevel)

	db, err := app.InitDB()
	if err != nil {
		log.Fatal(err)
	}

	stream, err := InitKinesis(db)
	if err != nil {
		log.Fatal(err)
	}

	log.Fatal(<-stream.Run())
}

func InitKinesis(db *model.RedisConn) (*events.Stream, error) {
	awsCreds := credentials.NewSharedCredentials("/home/cvp/.aws/credentials", "twitch-esports-aws")
	awsConfig := aws.NewConfig().WithRegion("us-west-2").WithCredentials(awsCreds)
	awsSession, err := session.NewSession(awsConfig)
	if err != nil {
		return nil, fmt.Errorf("error initializing AWS session: %s", err)
	}

	procChan := make(chan int, 1000)
	procCheck := make(chan struct{})
	go func() {
		for {
			procCheck <- struct{}{}
			time.Sleep(500 * time.Millisecond)
		}
	}()
	go func() {
		procCount := 0
		for {
			select {
			case delta := <-procChan:
				procCount += delta
			case <-procCheck:
				// fmt.Printf("\rmw events in queue: %-4d", procCount)
				// os.Stdout.Sync()
				if procCount > 1000 {
					log.Fatalf("too many pending events: %d", procCount)
				}
			}
		}
	}()

	mwLog := log.WithField("event", "minute watched")
	handleEvent := func(event events.MinuteWatchedEvent) error {
		if event.UserID == "" {
			return nil
		}
		mwUserLog := mwLog.WithField("user_id", event.UserID)

		procChan <- 1
		user, unlock, err := db.AddUserXP(event.UserID, 1, 45*time.Second, nil)
		if err != nil {
			mwUserLog.Errorf("error adding XP: %s", err)
			return err
		} else if user.Badge == 0 {
			mwUserLog = mwUserLog.WithField("badge_id", nil)
		} else {
			mwUserLog = mwUserLog.WithFields(log.Fields{
				"badge_id": user.Badge,
				"xp":       user.XP,
				"max_xp":   user.MaxXP,
			})
		}
		procChan <- -1
		mwUserLog.Debug()
		if len(unlock) > 0 {
			mwUserLog.WithField("unlocked", unlock).Debug("user unlocked new badges")
		}
		return nil
	}

	return events.InitKinesis(awsSession, handleEvent)
}
