package events

import (
	"encoding/json"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/content/evo/util"
)

type MinuteWatchedEvent struct {
	Channel       string
	ChannelID     int
	Live          bool
	MinutesLogged int
	UserID        string
	Subscriber    bool
	Time          time.Time
}

type MWEventFunc func(MinuteWatchedEvent) error

var mwRequiredFields []string
var pdt *time.Location

func init() {
	mwRequiredFields = []string{
		"channel",
		"channel_id",
		"live",
		"minutes_logged",
		"subscriber_web",
		"time",
		"user_id",
	}
	var err error
	if pdt, err = time.LoadLocation("US/Pacific"); err != nil {
		panic(fmt.Sprintf("error loading US/Pacific timezone: %s", err))
	}
}

const kinesisTimestamp = "2006-01-02 15:04:05"

// Fields are nested and all stringified, which makes parsing the idiomatic way (reflection on struct tags)
// super annoying / error prone. So, fuck it, do it manually, and that'll let us do more validation anyway.
func (e *MinuteWatchedEvent) UnmarshalJSON(data []byte) (err error) {
	var event map[string]interface{}
	if err := json.Unmarshal(data, &event); err != nil {
		return err
	} else if nameI, found := event["Name"]; !found {
		return errors.New("event has no Name")
	} else if name, ok := nameI.(string); !ok {
		return fmt.Errorf("event Name is not a string (%T): %#v", nameI, nameI)
	} else if name != "minute-watched" {
		return fmt.Errorf("event is not for a minute watched: %q", name)
	}

	var fields map[string]string
	if fieldsI, found := event["Fields"]; !found {
		return errors.New("event has no Fields")
	} else if fieldsM, ok := fieldsI.(map[string]interface{}); !ok {
		return fmt.Errorf("Fields is not a map (%T): %#v", fieldsI, fieldsI)
	} else {
		fields = make(map[string]string, len(fieldsM))
		for k, v := range fieldsM {
			if vStr, ok := v.(string); !ok {
				return fmt.Errorf("field %q does not have a string value (%T): %#v", k, v, v)
			} else {
				fields[k] = vStr
			}
		}
	}

	var missing []string
	for _, field := range mwRequiredFields {
		if _, found := fields[field]; !found {
			missing = append(missing, field)
		}
	}
	if len(missing) > 0 {
		return fmt.Errorf("missing fields: %s", strings.Join(missing, ", "))
	}

	// Fields storing non-string values can come in as the empty string.
	// For some fields, that's okay (minutes_logged, live, subscriber_web).
	if e.ChannelID, err = strconv.Atoi(fields["channel_id"]); err != nil {
		return fmt.Errorf("channel_id is not an integer: %q; event: %#v", fields["channel_id"], fields)
	} else if e.MinutesLogged, err = util.Atoi(fields["minutes_logged"], 0); err != nil {
		return fmt.Errorf("minutes_logged is not an integer: %q; event: %#v", fields["minutes_logged"], fields)
	} else if e.Live, err = util.Atob(fields["live"], false); err != nil {
		return fmt.Errorf("live is not a boolean: %q; event: %#v", fields["live"], fields)
	} else if e.Subscriber, err = util.Atob(fields["subscriber_web"], false); err != nil {
		return fmt.Errorf("subscriber_web is not a boolean: %q; event: %#v", fields["subscriber_web"], fields)
	} else if e.Time, err = time.ParseInLocation(kinesisTimestamp, fields["time"], pdt); err != nil {
		return fmt.Errorf("failed to parse time (expected \"YYYY-MM-DD HH:MM:SS\"): %q; events: %#v", fields["time"], fields)
	} else {
		e.Time = e.Time.UTC()
	}

	// string fields
	e.Channel = fields["channel"]
	e.UserID = fields["user_id"]

	return nil
}
