package events

import (
	"bytes"
	"compress/flate"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"

	"github.com/TwitchScience/kinsumer"
	"github.com/aws/aws-sdk-go/aws/session"
	log "github.com/sirupsen/logrus"
	"github.com/twinj/uuid"
)

const (
	// CLIENT_NAME = "evo-backend"
	CLIENT_NAME = "spade-downstream-prod-pc-mw"
	STREAM_ARN  = "arn:aws:kinesis:us-west-2:954268612818:stream/spade-downstream-prod-pc-mw"
	STREAM_NAME = "spade-downstream-prod-pc-mw"
)

type Stream struct {
	*kinsumer.Kinsumer
	stop      bool
	errs      chan error
	eventFunc func(MinuteWatchedEvent) error
	logger    *log.Entry
}

func InitKinesis(awsSession *session.Session, eventFunc MWEventFunc) (*Stream, error) {
	k, err := kinsumer.NewWithSession(awsSession, STREAM_NAME, CLIENT_NAME, uuid.NewV4().String(), kinsumer.NewConfig())
	if err != nil {
		return nil, err
	}
	return &Stream{
		Kinsumer:  k,
		stop:      false,
		errs:      make(chan error, 1),
		eventFunc: eventFunc,
		logger:    log.WithFields(log.Fields{"context": "kinesis"}),
	}, nil
}

func (s Stream) Run() <-chan error {
	if s.stop {
		s.logger.Error("Run() called on stopped Stream")
		s.errs <- errors.New("stopped")
	} else if err := s.Kinsumer.Run(); err != nil {
		s.logger.Errorf("error starting consumer: %s", err)
		s.errs <- fmt.Errorf("error starting kinsumer: %s", err)
	} else {
		s.logger.Info("started Kinesis consumer")
		go s.processEvents()
	}
	return s.errs
}

func (s Stream) Stop() {
	s.logger.Info("stopping consumer")
	s.stop = true
	s.Kinsumer.Stop()
}

func (s Stream) processEvents() {
	for !s.stop {
		data, err := s.Kinsumer.Next()
		if err != nil || data == nil {
			s.stop = true
			if err == nil {
				err = errors.New("stopped")
			}
			s.errs <- err
			return
		}
		go s.processEvent(data)
	}
}

type kinesisEvent struct {
	Version   int    `json:"Version"`
	UUID      string `json:"UUID"`
	CreatedAt string `json:"CreatedAt"`
	Data      []byte `json:"Data"`
}

func (s Stream) processEvent(data []byte) {
	// Kinesis events come as JSONs with fields: "Version", "UUID", "CreatedAt", and "Data".
	// We only care about "Data", which is base64-encoded and zlib-deflated with a window of -15,
	// and with a prefix byte that's there just to drive cvp nuts.

	// `encoding/json` does base64 decoding when unmarshaling into `[]byte`.
	var kEvent kinesisEvent
	var eventData []byte
	var mwEvents []MinuteWatchedEvent
	var err error
	if err = json.Unmarshal(data, &kEvent); err != nil {
		err = fmt.Errorf("error parsing raw Kinesis event: %s", err)
	} else if eventData, err = ioutil.ReadAll(flate.NewReader(bytes.NewBuffer(kEvent.Data[1:]))); err != nil {
		err = fmt.Errorf("error decompressing event data: %s", err)
	} else if err = json.Unmarshal(eventData, &mwEvents); err != nil {
		err = fmt.Errorf("error parsing event data: %s", err)
	} else {
		for _, mwEvent := range mwEvents {
			if err = s.eventFunc(mwEvent); err != nil {
				err = fmt.Errorf("event processing failed: %s; event: %#v", err, mwEvent)
				break
			}
		}
	}

	// For now, crash on any error.
	if err != nil {
		s.logger.Error(err.Error())
		s.stop = true
		s.errs <- err
	}
}
