package model

import (
	"database/sql"
	"fmt"
)

// TODO: unlocks

type Evolution struct {
	ID              int                   `db:"id"    json:"id"`
	Name            string                `db:"name"  json:"name"`
	EvolutionSetIDs []int                 `db:"-"     json:"evolution_set_ids"`
	EvolutionSets   map[int]*EvolutionSet `db:"-"     json:"-"`
}

type EvolutionSet struct {
	ID          int            `db:"id"            json:"id"`
	Name        string         `db:"name"          json:"name"`
	XPCap       int            `db:"xp_cap"        json:"xp_cap"`
	EvolutionID int            `db:"evolution_id"  json:"evolution_id"`
	Evolution   *Evolution     `db:"-"             json:"-"`
	BadgeIDs    []int          `db:"-"             json:"badge_ids"`
	Badges      map[int]*Badge `db:"-"             json:"-"`
}

type Badge struct {
	ID             int           `db:"id"                json:"id"`
	EvolutionSetID int           `db:"evolution_set_id"  json:"evolution_set_id"`
	EvolutionSet   *EvolutionSet `db:"-"                 json:"-"`
	EvolutionID    int           `db:"evolution_id"      json:"evolution_id"`
	XPRequired     string        `db:"xp_required"       json:"xp_required"`
	Name           string        `db:"name"              json:"name"`
	ImageURL       string        `db:"image_url"         json:"image_url"`
}

func (db *EvoDB) syncNames(table string) (map[int]string, error) {
	rows, err := db.conn.Queryx(`SELECT "id", "name" FROM ?`, table)
	if err == sql.ErrNoRows {
		return nil, fmt.Errorf("database has not been seeded")
	} else if err != nil {
		return nil, err
	}
	defer rows.Close()

	names := make(map[int]string)
	for rows.Next() {
		var id int
		var name string
		if err = rows.Scan(&id, &name); err != nil {
			return nil, err
		}
		names[id] = name
	}
	return names, nil
}

func (db *EvoDB) SyncEvoCache() error {
	// "evolutions" table
	var evolutions []Evolution
	if err := db.conn.Select(&evolutions, `SELECT * FROM "evolutions"`); err == sql.ErrNoRows {
		return fmt.Errorf("database has not been seeded")
	} else if err != nil {
		return fmt.Errorf("error syncing evolutions: %s", err)
	}
	db.evoMap = make(map[int]*Evolution, len(evolutions))
	for _, evo := range evolutions {
		evo.EvolutionSets = make(map[int]*EvolutionSet)
		db.evoMap[evo.ID] = &evo
	}

	// "evolution_sets" table
	var evolutionSets []EvolutionSet
	if err := db.conn.Select(&evolutions, `SELECT * FROM "evolution_sets"`); err == sql.ErrNoRows {
		return fmt.Errorf("database has not been seeded")
	} else if err != nil {
		return fmt.Errorf("error syncing evolution sets: %s", err)
	}
	db.evoSets = make(map[int]*EvolutionSet, len(evolutionSets))
	for _, evoSet := range evolutionSets {
		evoSet.Badges = make(map[int]*Badge)
		evo, found := db.evoMap[evoSet.EvolutionID]
		if !found {
			return fmt.Errorf("found evolution set (%d) for non-existent evolution (%d)", evoSet.ID, evoSet.EvolutionID)
		}
		evoSet.Evolution = evo
		evo.EvolutionSetIDs = append(evo.EvolutionSetIDs, evoSet.ID)
		evo.EvolutionSets[evoSet.ID] = &evoSet
		db.evoSets[evoSet.ID] = &evoSet
	}

	// "badges" table
	var badges []Badge
	if err := db.conn.Select(&badges, `SELECT * FROM "badges"`); err == sql.ErrNoRows {
		return fmt.Errorf("database has not been seeded")
	} else if err != nil {
		return fmt.Errorf("error syncing badges: %s", err)
	}
	for _, badge := range badges {
		evoSet, found := db.evoSets[badge.EvolutionSetID]
		if !found {
			return fmt.Errorf("found badge (%d) for non-existent evolution set (%d)", badge.ID, badge.EvolutionSetID)
		}
		badge.EvolutionSet = evoSet
		evoSet.BadgeIDs = append(evoSet.BadgeIDs, badge.ID)
		evoSet.Badges[badge.ID] = &badge
	}

	return nil
}

func (db *EvoDB) ListEvolutions() ([]*Evolution, error) {
	evolutions := make([]*Evolution, 0, len(db.evoMap))
	for _, evo := range db.evoMap {
		evolutions = append(evolutions, evo)
	}
	return evolutions, nil
}

func (db *EvoDB) FindEvolution(id int) (*Evolution, error) {
	if evo, found := db.evoMap[id]; !found {
		return nil, fmt.Errorf("no such evolution: %d", id)
	} else {
		return evo, nil
	}
}

func (db *EvoDB) ListEvolutionSets(evolutionID int) ([]*EvolutionSet, error) {
	evo, err := db.FindEvolution(evolutionID)
	if err != nil {
		return nil, err
	}
	evolutionSets := make([]*EvolutionSet, 0, len(evo.EvolutionSets))
	for _, evoSet := range evo.EvolutionSets {
		evolutionSets = append(evolutionSets, evoSet)
	}
	return evolutionSets, nil
}

func (db *EvoDB) FindEvolutionSet(id int) (*EvolutionSet, error) {
	if evoSet, found := db.evoSets[id]; !found {
		return nil, fmt.Errorf("no such evolution set: %d", id)
	} else {
		return evoSet, nil
	}
}
