/*************************************************
 * Static metadata about evolutions/sets/badges. *
 *************************************************/

/* These tables will more than likely be small enough, and referenced often enough,
 * that an application should probably just do a full table scan and cache on init.
 */

CREATE TABLE IF NOT EXISTS "evolutions" (
    "id"    SERIAL PRIMARY KEY,
    "name"  TEXT NOT NULL
);

CREATE TABLE IF NOT EXISTS "evolution_sets" (
    "id"            SERIAL PRIMARY KEY,
    "evolution_id"  INTEGER NOT NULL,
    "xp_cap"        INTEGER NOT NULL,
    "name"          TEXT NOT NULL
);

CREATE TABLE IF NOT EXISTS "badges" (
    "id"                SERIAL PRIMARY KEY,
    "evolution_id"      INTEGER NOT NULL,
    "evolution_set_id"  INTEGER NOT NULL,
    "xp_required"       INTEGER NOT NULL,
    "name"              TEXT NOT NULL,
    "image_url"         TEXT
);
CREATE UNIQUE INDEX IF NOT EXISTS "badges_by_evolution_id" ON "badges" ("evolution_id");
CREATE UNIQUE INDEX IF NOT EXISTS "badges_by_evolution_set_id" ON "badges" ("evolution_set_id");

CREATE TABLE IF NOT EXISTS "evolution_unlocks" (
    "evolution_id"      INTEGER NOT NULL,
    "required_set_ids"  INTEGER[],
    "required_user_xp"  INTEGER,
    "unlock_set_ids"    INTEGER[] NOT NULL
);


/**************
 * User data. *
 **************/

/* XP tracking is divided across a couple of tables:
 *
 * "users"
 *     A user's currently active evolution set has XP tracked within the user record itself.
 *     Most user queries will care about active XP already anyway (-1 DB op for an extremely common query),
 *     and this also (hopefully) allows an atomic XP increment to validate against "last_xp_update" in the same op.
 *
 * "user_xp"
 *     XP amounts for non-active evolution sets.
 *     If the user has an active evolution set, its record in this table will probably be stale, if it exists at all.
 *     If this record does not exist, it means 0 XP. Try to avoid adding a row before XP is gained.
 *     This table knows nothing about which badges/sets are _available_ to a user; check "user_badges" first.
 *
 * "user_xp_log"
 *     Log of XP amounts per user and evolution set at the time a user changed their active set.
 *     Used for XP auditing / reconciliation.
 */

CREATE TABLE IF NOT EXISTS "users" (
    "id"                       SERIAL PRIMARY KEY,
    "opaque_id"                TEXT,
    "twitch_id"                TEXT,
    "xp_total"                 INTEGER NOT NULL DEFAULT 0,

    "active_evolution_id"      INTEGER NOT NULL DEFAULT 0,
    "active_evolution_set_id"  INTEGER NOT NULL DEFAULT 0,
    "active_since"             TIMESTAMP,

    "active_xp"                INTEGER NOT NULL DEFAULT 0,
    "active_xp_cap"            INTEGER NOT NULL DEFAULT 0,
    "last_xp_update"           TIMESTAMP
);
CREATE UNIQUE INDEX IF NOT EXISTS "users_by_opaque_id" ON "users" ("opaque_id");
CREATE UNIQUE INDEX IF NOT EXISTS "users_by_twitch_id" ON "users" ("twitch_id");

CREATE TABLE IF NOT EXISTS "user_xp" (
    "user_id"           INTEGER,
    "evolution_set_id"  INTEGER,
    "xp"                INTEGER NOT NULL DEFAULT 0,
    PRIMARY KEY ("user_id", "evolution_set_id")
);

CREATE TABLE IF NOT EXISTS "user_xp_log" (
    "user_id"           INTEGER NOT NULL,
    "evolution_set_id"  INTEGER NOT NULL,
    "xp"                INTEGER NOT NULL,
    "since"             TIMESTAMP NOT NULL,
    "until"             TIMESTAMP NOT NULL DEFAULT NOW(),
    PRIMARY KEY ("user_id", "until")
);

/* Available and finished badges. */
CREATE TABLE IF NOT EXISTS "user_evolution_sets" (
    "user_id"                      INTEGER,
    "evolution_id"                 INTEGER,
    "finished_evolution_set_ids"   INTEGER[] NOT NULL DEFAULT '{}',
    "available_evolution_set_ids"  INTEGER[] NOT NULL DEFAULT '{}',
    PRIMARY KEY ("user_id", "evolution_id")
);
/* GIN indexes are recommended for fast set operations:
 * https://www.postgresql.org/docs/10/static/indexes-types.html */
CREATE INDEX IF NOT EXISTS "finished_user_evolution_sets" ON "user_evolution_sets" USING GIN ("finished_evolution_set_ids");
