package model

import (
	"database/sql"
	"errors"
	"fmt"
	"time"
)

type User struct {
	ID                   int       `db:"id"                       json:"id"`
	OpaqueID             string    `db:"opaque_id"                json:"opaque_id"`
	TwitchID             string    `db:"twitch_id"                json:"twitch_id"`
	XPTotal              int       `db:"xp_total"                 json:"xp_total"`
	ActiveEvolutionSetID int       `db:"active_evolution_set_id"  json:"active_evolution_set_id"`
	ActiveSince          time.Time `db:"active_since"             json:"-"`
	ActiveXP             int       `db:"active_xp"                json:"active_xp"`
	ActiveXPCap          int       `db:"active_xp_cap"            json:"active_xp_cap"`

	db *EvoDB
}

type UserEvolutionSet struct {
	UserID         int   `db:"user_id"           json:"user_id"`
	User           *User `db:"-"                 json:"-"`
	EvolutionID    int   `db:"-"                 json:"evolution_id"`
	EvolutionSetID int   `db:"evolution_set_id"  json:"evolution_set_id"`
	CurrentXP      int   `db:"xp"                json:"xp"`
	XPCap          int   `db:"-"                 json:"xp_cap"`
	Active         bool  `db:"-"                 json:"active"`
}

// Find & Create

func (db *EvoDB) FindUserByID(id int) (*User, error) {
	var user User
	if err := db.conn.Get(&user, `SELECT * FROM "users" WHERE "id" = ? LIMIT 1`, id); err == sql.ErrNoRows {
		return nil, nil
	} else if err != nil {
		return nil, err
	} else {
		return &user, err
	}
}

func (db *EvoDB) FindUserByOpaqueID(id string) (*User, error) {
	var user User
	if err := db.conn.Get(&user, `SELECT * FROM "users" WHERE "opaque_id" = ? LIMIT 1`, id); err == sql.ErrNoRows {
		return nil, nil
	} else if err != nil {
		return nil, err
	} else {
		return &user, nil
	}
}

func (db *EvoDB) FindOrCreateUser(opaqueID string) (*User, error) {
	var user User
	if err := db.conn.Get(&user,
		`INSERT INTO "users" ("opaque_id", "twitch_id") VALUES (?, ?) ON CONFLICT DO NOTHING RETURNING *`,
		opaqueID, "twitch:"+opaqueID, // TODO: get actual Twitch ID, lol
	); err != nil {
		return nil, err
	} else {
		return &user, nil
	}
}

// Getters

func (user *User) ActiveEvolutionSet() (*UserEvolutionSet, error) {
	if user.ActiveEvolutionSetID == 0 {
		return nil, nil
	}
	return user.FindEvolutionSet(user.ActiveEvolutionSetID)
}

func (user *User) FindEvolutionSet(id int) (*UserEvolutionSet, error) {
	evoSet, found := user.db.evoSets[id]
	if !found {
		return nil, fmt.Errorf("no such evolution set: %d", id)
	}

	var xp int
	if id == user.ActiveEvolutionSetID {
		xp = user.ActiveXP
	} else {
		if err := user.db.conn.Get(&xp,
			`SELECT "xp" FROM "user_xp" WHERE "user_id" = ? AND "evolution_set_id" = ? LIMIT 1`,
			user.ID, id,
		); err != nil && err != sql.ErrNoRows {
			return nil, fmt.Errorf("error getting user %q xp for evolution set %d: %s", user.ID, id, err)
		}
	}

	return &UserEvolutionSet{
		UserID:         user.ID,
		User:           user,
		EvolutionID:    evoSet.EvolutionID,
		EvolutionSetID: id,
		CurrentXP:      xp,
		XPCap:          evoSet.XPCap,
		Active:         id == user.ActiveEvolutionSetID,
	}, nil
}

func (user *User) AvailableEvolutionSetIDs(evoID int) (ids []int, err error) {
	if err := user.db.conn.Get(&ids,
		`SELECT "available_evolution_set_ids" FROM "user_evolution_sets" WHERE "user_id" = ? AND "evolution_id" = ?`,
		user.ID, evoID,
	); err == sql.ErrNoRows {
		// TODO: return the base available set for this evolution (don't insert new row until user has some xp, though)
		return nil, nil
	}
	return
}

func (user *User) AvailableEvolutionSets(evoID int) ([]*UserEvolutionSet, error) {
	evoSetIDs, err := user.AvailableEvolutionSetIDs(evoID)
	if err != nil {
		return nil, err
	}

	evoSets := make([]*UserEvolutionSet, len(evoSetIDs))
	for i, id := range evoSetIDs {
		if evoSets[i], err = user.FindEvolutionSet(id); err != nil {
			return nil, err
		}
	}
	return evoSets, nil
}

// Update

func (user *User) UpdateEvolutionSet(id int) (*UserEvolutionSet, error) {
	if user.ActiveEvolutionSetID == id {
		return user.ActiveEvolutionSet()
	}

	// TODO: check if in available set

	userEvoSet, err := user.FindEvolutionSet(id)
	if err != nil {
		return nil, err
	}

	tx, err := user.db.conn.Beginx()
	if err != nil {
		return nil, fmt.Errorf("error initiating transaction: %s", err)
	}
	defer tx.Rollback()

	// Save current active XP to "user_xp" and "user_xp_log".
	if user.ActiveEvolutionSetID != 0 {
		if _, err = tx.Exec(
			`INSERT INTO "user_xp" ("user_id", "evolution_set_id", "xp") VALUES (?, ?, ?) ON CONFLICT SET "xp" = ?`,
			user.ID, user.ActiveEvolutionSetID, user.ActiveXP, user.ActiveXP,
		); err != nil {
			return nil, fmt.Errorf("error saving xp for last active badge: %s", err)
		}

		if _, err = tx.Exec(
			`INSERT INTO "user_xp_log" ("user_id", "evolution_set_id", "xp", "since") VALUES (?, ?, ?, ?)`,
			user.ID, user.ActiveEvolutionSetID, user.ActiveXP, user.ActiveSince,
		); err != nil {
			return nil, fmt.Errorf("error updating transaction log: %s", err)
		}
	}

	// Set new "active" fields.
	activeSince := time.Now()
	if _, err = tx.Exec(
		`UPDATE "users" SET ("active_evolution_id", "active_evolution_set_id", "active_since", "active_xp", "active_xp_cap") = (?, ?, ?, ?, ?) WHERE "id" = ?`,
		userEvoSet.EvolutionID, id, activeSince, userEvoSet.CurrentXP, userEvoSet.XPCap, user.ID,
	); err != nil {
		return nil, fmt.Errorf("error updating active evolution set id: %s", err)
	}

	if err = tx.Commit(); err != nil {
		return nil, fmt.Errorf("error updating active evolution set: %s", err)
	}

	user.ActiveEvolutionSetID = id
	user.ActiveSince = activeSince
	user.ActiveXP = userEvoSet.CurrentXP
	user.ActiveXPCap = userEvoSet.XPCap
	userEvoSet.Active = true
	return userEvoSet, nil
}

func (user *User) AddXP() error {
	return errors.New("not yet implemented")
}
