// +build test

package util

import (
	"bytes"
	"fmt"
	"reflect"
)

// Returned by `StructDiff`.
type DiffField struct {
	Name string
	A, B interface{}
}

// Like `reflect.DeepEqual`, but accepts only structs and describes differences.
func StructDiff(aArg, bArg interface{}) (diffs []DiffField) {
	a := reflect.ValueOf(aArg)
	b := reflect.ValueOf(bArg)
	if a.Type() != b.Type() {
		panic(fmt.Sprintf("arguments are not the same type: %T vs %T", aArg, bArg))
	} else if a.Kind() != reflect.Struct {
		panic(fmt.Sprintf("arguments are not structs: %T", aArg))
	}

	t := a.Type()
	for i := 0; i < t.NumField(); i++ {
		field := t.Field(i)
		aValue := a.Field(i).Interface()
		bValue := b.Field(i).Interface()
		if !reflect.DeepEqual(aValue, bValue) {
			diffs = append(diffs, DiffField{field.Name, aValue, bValue})
		}
	}
	return diffs
}

// Convenience method that formats a test failure message based on `StructDiff`'s output.
func DiffErrorMessage(diffs []DiffField) string {
	buf := bytes.NewBufferString("structs differ:\n")
	for _, diff := range diffs {
		buf.WriteString(fmt.Sprintf("    %q (%T): %#v vs %#v\n", diff.Name, diff.A, diff.A, diff.B))
	}
	return buf.String()
}
