---
title: GraphQL Schema
weight: 2
---

The activity feed produced by Sauron is exposed via GQL as a paginated query so that clients can load and browse
historical events. It provides up to 28 days of history.

## Schema

```gql
# Represents the activity feed for a channel. Includes information on events in the feed and available filters.
type ActivityFeed {
  # Paginated past activities for the activity feed.
  # Always ordered by descending timestamp.
  events(first: Int = 10, after: Cursor): ActivityFeedConnection

  # List of filter categories.
  filterCategories: [ActivityFeedFilterCategory!]
}

# Represents a filter category for activity feed events.
type ActivityFeedFilterCategory {
  # ID of the filter category.
  id: ID!

  # Localized display name for the filter category.
  displayName: String!
}

# A paginated list of activity feed events.
type ActivityFeedConnection {
  # The list of activity feed events on this page.
  edges: [ActivityFeedEdge!]!

  # Information about this page of activity feed events.
  pageInfo: PageInfo!
}

# An activity feed event edge composed of a cursor and node.
type ActivityFeedEdge {
  # The cursor of the record, opaque to the user.
  cursor: Cursor!

  # The node of the nedge contains all information about an activity feed event.
  node: ActivityFeedEvent!
}

# An activity feed event representing an event that happened in a creator's channel.
type ActivityFeedEvent {
  # Unique ID of the event.
  id: ID!

  # The type of event being handled.
  type: ActivityFeedEventType!

  # ID of the filter category this event belongs to.
  filterCategoryID: ID!

  # The timestamp of when the event occurred.
  timestamp: Time!

  # Optional localized title content for the event.
  title: ActivityFeedText

  # Optional localized content for the event.
  content: ActivityFeedText

  # Optional additional localized content for the body of an event.
  contentBody: ActivityFeedText

  # User that triggered the event.
  user: User

  # Optional user provided content for the event.
  userContent: ActivityFeedText
}

# The type of activity event that ocurred. DEFAULT is the fallback for new event types.
enum ActivityFeedEventType {
  # Default event type, used as a fallback for events that don't have a type assigned.
  DEFAULT

  # Event represents a follow.
  FOLLOW

  # Event represents a host.
  HOST

  # Event represents a raid.
  RAID

  # Event represents an autohost.
  AUTOHOST

  # Event represents the usage of bits (a cheer) by a user.
  BITS_USAGE

  # Event represents a new subscription from a user.
  SUBSCRIPTION

  # Event represents a resubscription by a user.
  RESUBSCRIPTION

  # Event represents a new Prime subscription by a user.
  PRIME_SUBSCRIPTION

  # Event represents a Prime resubscription by a user.
  PRIME_RESUBSCRIPTION

  # A subscription was gifted by an individual user to another user.
  INDIVIDUAL_GIFT_SUBSCRIPTION

  # One ore more subscriptions were gifted to the community by a user.
  COMMUNITY_GIFT_SUBSCRIPTION

  # A hype train has started.
  HYPE_TRAIN

  # A channel points reward was redeemed.
  CHANNEL_POINTS_REWARD_REDEMPTION

  # A community goal has been completed.
  COMMUNITY_GOAL_COMPLETE

  # A boost has started.
  BOOST_START

  # A boot has been completed.
  BOOST_COMPLETE

  # The drop claim window has opened.
  DROP_CLAIM_WINDOW_OPEN
}

# Localized text for the contents of an activity feed event. Contains individual fragments to be rendered and fallback text for clients that cannot render fragments.
type ActivityFeedText {
  # A list of text fragments for this content.
  fragments: [ActivityFeedTextFragment!]!

  # Fallback text for an event.
  fallbackText: String!
}

# A fragment of localized text made up of a token and display properties for the text.
type ActivityFeedTextFragment {
  # A token to be rendered by a client.
  token: ActivityFeedToken!

  # The token should be rendered with emphasis.
  hasEmphasis: Boolean!
}

# A token representing text.
type ActivityFeedTextToken {
  # The text of this token.
  text: String!
}

# A token representing an integer value.
type ActivityFeedIntegerToken {
  # The integer value of this token.
  number: Int!
}

# A token representing a percentage value.
type ActivityFeedPercentToken {
  # The percent value of this token represented as [0, 1].
  percent: Float!
}

# A union of tokens, which can be text, an integer, a percent value, or details about a user.
union ActivityFeedToken =
    ActivityFeedTextToken
  | ActivityFeedIntegerToken
  | ActivityFeedPercentToken
  | User
  | Emote
```

## Example Queries

### Retrieve list of filter categories
```gql
query {
  channel(id: "40972890") {
    name
    activityFeed {
      filterCategories {
        id
        displayName
      }
    }
  }
}
```

### Retrieve activities for user ID
```gql
fragment Text on ActivityFeedText {
   fragments {
    hasEmphasis
    token {
      __typename
      ...on ActivityFeedTextToken {
        text
      }
      ...on ActivityFeedIntegerToken {
        number
      }
      ...on ActivityFeedPercentToken {
        percent
      }
      ...on Emote {
        id
        setID
      }
      ...on User {
        displayName
      }
    }
  } 
}

query {
  channel(id: "40972890") {
    name
    activityFeed {
       events(first: 100) {
        edges {
          node {
            id
            filterCategoryID
            title {
              ...Text
            }
            content {
              ...Text
            }
            contentBody {
              ...Text
            }
            userContent {
              ...Text
            }
          }
        }
      }
    }
  }
}
```